package com.finconsgroup.itserr.marketplace.usercommunication.dm.exception;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.messaging.MessagingException;
import org.springframework.messaging.handler.annotation.MessageExceptionHandler;
import org.springframework.messaging.handler.annotation.support.MethodArgumentNotValidException;
import org.springframework.messaging.simp.SimpMessageHeaderAccessor;
import org.springframework.messaging.simp.annotation.SendToUser;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;

import java.util.LinkedList;
import java.util.List;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageHeaders.USER_DESTINATION_DEFAULTS;
import static org.springframework.http.HttpStatus.BAD_REQUEST;
import static org.springframework.http.HttpStatus.INTERNAL_SERVER_ERROR;

@ControllerAdvice
@Slf4j
public class MessagingExceptionHandler {

    private static final String VALIDATION_ERROR_ARGUMENT_SEPARATOR = ": ";

    private static final String LOG_ERROR_MESSAGE = "Error occurred for request: %s";

    /**
     * Handles exceptions thrown when validation errors on method arguments occur.
     *
     * @param ex             the exception thrown, the type of exception is {@link MethodArgumentNotValidException}
     * @param headerAccessor the header accessor
     * @return a response dto with bad request status and validation error messages
     */
    @SendToUser("/queue/errors")
    @MessageExceptionHandler({MethodArgumentNotValidException.class})
    public MessagingErrorResponseDto handleMethodArgumentNotValidException(
            final MethodArgumentNotValidException ex, final SimpMessageHeaderAccessor headerAccessor
    ) {
        logError(ex, headerAccessor);
        List<String> errors = new LinkedList<>();
        if (ex.getBindingResult() != null) {
            ex.getBindingResult().getFieldErrors().stream()
                    .map(fieldError -> {
                        String fieldName = fieldError.getField();
                        String errorMessage = fieldError.getDefaultMessage();
                        return fieldName + VALIDATION_ERROR_ARGUMENT_SEPARATOR + errorMessage;
                    })
                    .forEach(errors::add);
            ex.getBindingResult().getGlobalErrors().stream()
                    .map(globalError -> {
                        String fieldName = globalError.getObjectName();
                        String errorMessage = globalError.getDefaultMessage();
                        return fieldName + VALIDATION_ERROR_ARGUMENT_SEPARATOR + errorMessage;
                    })
                    .forEach(errors::add);
        } else {
            errors.add(ex.getMethodParameter().getParameterName() + VALIDATION_ERROR_ARGUMENT_SEPARATOR +
                    ex.getMessage());
        }
        return createError(BAD_REQUEST, errors, ex, headerAccessor);
    }

    /**
     * Handles generic messaging exceptions.
     *
     * @param ex      the exception thrown, the type of exception is {@link MessagingException}
     * @param headerAccessor the header accessor
     * @return a response dto with an internal server error status and empty message
     */
    @SendToUser("/queue/errors")
    @ExceptionHandler({MessagingException.class})
    public MessagingErrorResponseDto handleMessagingException(final MessagingException ex,
                                                              final SimpMessageHeaderAccessor headerAccessor) {
        logError(ex, headerAccessor);
        return createError(INTERNAL_SERVER_ERROR, "", ex, headerAccessor);
    }

    /**
     * Logs the given exception with a formatted error message.
     *
     * @param ex             the exception to log
     * @param headerAccessor the header accessor
     */
    public void logError(Exception ex, SimpMessageHeaderAccessor headerAccessor) {
        String errorMessage = LOG_ERROR_MESSAGE.formatted(headerAccessor);
        log.error(errorMessage, ex);
    }

    /**
     * Creates a {@link ResponseEntity} containing a single error message.
     *
     * @param httpStatus     the HTTP status to set in the response
     * @param message        the error message to include
     * @param ex             the thrown exception
     * @param headerAccessor the header accessor
     * @return a response entity with the specified HTTP status and error message
     */
    public MessagingErrorResponseDto createError(
            HttpStatus httpStatus,
            String message,
            Exception ex,
            SimpMessageHeaderAccessor headerAccessor
    ) {
        return createError(httpStatus, List.of(message), ex, headerAccessor);
    }

    /**
     * Creates a {@link ResponseEntity} containing multiple error messages.
     *
     * @param httpStatus     the HTTP status to set in the response
     * @param messages       the list of error messages to include
     * @param ex             the thrown exception
     * @param headerAccessor the header accessor
     * @return a response dto with the specified HTTP status and error messages
     */
    public MessagingErrorResponseDto createError(
            HttpStatus httpStatus,
            List<String> messages,
            Exception ex,
            SimpMessageHeaderAccessor headerAccessor
    ) {
        String destination = headerAccessor.getDestination();
        String requestMessageId = headerAccessor.getFirstNativeHeader("requestMessageId");
        List<String> stackTrace = log.isDebugEnabled() ? ExceptionUtils.getRootCauseStackTraceList(ex) : null;
        USER_DESTINATION_DEFAULTS.forEach((k, v) -> headerAccessor.setNativeHeader(k, v.toString()));
        return new MessagingErrorResponseDto(httpStatus.value(), messages, destination, requestMessageId, stackTrace);
    }
}
