package com.finconsgroup.itserr.marketplace.usercommunication.dm.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.configuration.properties.WebSocketConfigurationProperties;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.StompOAuth2ClientCredentialsWithRefreshProvider;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.messaging.simp.config.MessageBrokerRegistry;
import org.springframework.messaging.simp.stomp.StompBrokerRelayMessageHandler;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.web.socket.config.annotation.EnableWebSocketMessageBroker;
import org.springframework.web.socket.config.annotation.StompEndpointRegistry;
import org.springframework.web.socket.config.annotation.WebSocketMessageBrokerConfigurer;

import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

/**
 * WebSocket and STOMP broker configuration for application and user destinations.
 */
@Configuration
@EnableWebSocketMessageBroker
@RequiredArgsConstructor
@Slf4j
public class WebSocketConfig implements WebSocketMessageBrokerConfigurer {

    private final WebSocketConfigurationProperties webSocketConfigurationProperties;
    private final ObjectMapper objectMapper;
    private final MessageDestinations messageDestinations;

    @Override
    public void configureMessageBroker(@NonNull MessageBrokerRegistry config) {
        // Enable a simple memory-based message broker to carry the greeting messages back to the client
        // on destinations prefixed with "/topic" and "/queue"
        // Send a heartbeat from the server every 10 seconds and expects a response from the client every 20 seconds

        // if broker relay is enabled then configure the same
        if (webSocketConfigurationProperties.isEnableBrokerRelay()) {
            config.enableStompBrokerRelay("/topic", "/queue")
                    .setSystemHeartbeatSendInterval(10_000L)
                    .setSystemHeartbeatReceiveInterval(20_000L)
                    .setTaskScheduler(heartBeatScheduler())
                    .setRelayHost(webSocketConfigurationProperties.getRelayHost())
                    .setVirtualHost(webSocketConfigurationProperties.getRelayVirtualHost())
                    .setRelayPort(webSocketConfigurationProperties.getRelayPort())
                    .setClientLogin(webSocketConfigurationProperties.getRelayClientLogin())
                    .setClientPasscode(webSocketConfigurationProperties.getRelayClientPassword())
                    .setSystemLogin(webSocketConfigurationProperties.getRelaySystemLogin())
                    .setSystemPasscode(webSocketConfigurationProperties.getRelaySystemPassword())
                    .setUserDestinationBroadcast(messageDestinations.getUserDestinationBroadcastTopic())
                    .setUserRegistryBroadcast(messageDestinations.getUserRegistryBroadcastTopic());
        } else {
            // otherwise use a simple broker
            config.enableSimpleBroker("/topic", "/queue")
                    .setHeartbeatValue(new long[]{10_000, 20_000})
                    .setTaskScheduler(heartBeatScheduler());
        }

        // Designate the "/app" prefix for messages that are bound for methods annotated with @MessageMapping
        config.setApplicationDestinationPrefixes("/app");

        // Set user destination prefix for private messages
        config.setUserDestinationPrefix(messageDestinations.getUserDestinationPrefix());
    }

    @Override
    public void registerStompEndpoints(StompEndpointRegistry registry) {
        // Register the "/sock-ws" endpoint, enabling SockJS fallback options so that
        // alternative transports may be used if WebSocket is not available
        registry.addEndpoint("/user-communication/sock-ws")
                .setAllowedOriginPatterns(webSocketConfigurationProperties.getCorsAllowedOrigins())
                .withSockJS();

        // Also register a plain WebSocket endpoint for clients that support it
        registry.addEndpoint("/user-communication/ws")
                .setAllowedOriginPatterns(webSocketConfigurationProperties.getCorsAllowedOrigins());

        // Custom error handler for returning proper error messages to the client
        registry.setErrorHandler(new WebSocketStompErrorHandler(objectMapper));
    }

    @Bean
    public TaskScheduler heartBeatScheduler() {
        return new ThreadPoolTaskScheduler();
    }

    @Bean(destroyMethod = "shutdown")
    public ScheduledExecutorService oauthTokenRefreshExecutor() {
        return Executors.newSingleThreadScheduledExecutor();
    }

    @Nullable
    @Bean
    public StompOAuth2ClientCredentialsWithRefreshProvider stompOAuth2ClientCredentialsWithRefreshProvider(
            Optional<StompBrokerRelayMessageHandler> stompBrokerRelayMessageHandlerOptional) {
        if (stompBrokerRelayMessageHandlerOptional.isPresent()
                && webSocketConfigurationProperties.getRelayOauthKeycloakTokenUrl() != null
                && !webSocketConfigurationProperties.getRelayOauthKeycloakTokenUrl().isEmpty()) {
            return new StompOAuth2ClientCredentialsWithRefreshProvider(
                    webSocketConfigurationProperties.getRelayOauthKeycloakTokenUrl(),
                    webSocketConfigurationProperties.getRelaySystemLogin(),
                    webSocketConfigurationProperties.getRelaySystemPassword(),
                    webSocketConfigurationProperties.getRelayOauthGrantType(),
                    webSocketConfigurationProperties.getRelayOauthParameters(),
                    webSocketConfigurationProperties.getRelayOauthRefreshDelay(),
                    webSocketConfigurationProperties.getRelayOauthRefreshPeriod(),
                    oauthTokenRefreshExecutor(),
                    stompBrokerRelayMessageHandlerOptional.get()
            );
        } else {
            return null;
        }
    }

}