package com.finconsgroup.itserr.marketplace.usercommunication.dm.config;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.configuration.properties.RabbitMQMessagingClientProperties;
import com.rabbitmq.client.impl.CredentialsProvider;
import com.rabbitmq.client.impl.CredentialsRefreshService;
import com.rabbitmq.client.impl.DefaultCredentialsProvider;
import com.rabbitmq.client.impl.DefaultCredentialsRefreshService;
import com.rabbitmq.client.impl.OAuth2ClientCredentialsGrantCredentialsProvider;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.core.QueueBuilder;
import org.springframework.amqp.core.TopicExchange;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.time.Duration;
import java.util.function.Function;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessagingType.RABBIT;

/**
 * RabbitMQ topology and template configuration for chats and notifications.
 */
@Configuration
@ConditionalOnProperty(value = "type", prefix = "user-communication.dm.messaging", havingValue = RABBIT)
public class RabbitMQConfig {


    @Bean
    public TopicExchange chatExchange(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return new TopicExchange(rabbitMQMessagingClientProperties.getChat().getExchange());
    }

    @Bean
    public Queue chatQueue(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return QueueBuilder.durable(rabbitMQMessagingClientProperties.getChat().getChatQueue()).build();
    }

    @Bean
    public Queue userNotificationQueue(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return QueueBuilder.durable(rabbitMQMessagingClientProperties.getChat().getUserNotificationQueue()).build();
    }

    @Bean
    public Binding chatBinding(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return BindingBuilder
                .bind(chatQueue(rabbitMQMessagingClientProperties))
                .to(chatExchange(rabbitMQMessagingClientProperties))
                .with(rabbitMQMessagingClientProperties.getChat().getChatRoutingKey());
    }

    @Bean
    public Binding userNotificationBinding(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return BindingBuilder
                .bind(userNotificationQueue(rabbitMQMessagingClientProperties))
                .to(chatExchange(rabbitMQMessagingClientProperties))
                .with(rabbitMQMessagingClientProperties.getChat().getUserNotificationRoutingKey());
    }

    @Bean
    public MessageConverter jsonMessageConverter() {
        return new Jackson2JsonMessageConverter();
    }

    @Bean
    public RabbitTemplate rabbitTemplate(ConnectionFactory connectionFactory) {
        RabbitTemplate rabbitTemplate = new RabbitTemplate(connectionFactory);
        rabbitTemplate.setMessageConverter(jsonMessageConverter());
        return rabbitTemplate;
    }

    @Bean
    public CredentialsProvider credentialsProvider(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        if (StringUtils.isBlank(rabbitMQMessagingClientProperties.getKeycloakTokenUrl())) {
            return createDefaultCredentialsProvider(rabbitMQMessagingClientProperties);
        } else {
            return createOAuthCredentialsProvider(rabbitMQMessagingClientProperties);
        }
    }

    private CredentialsProvider createDefaultCredentialsProvider(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        return new DefaultCredentialsProvider(
                rabbitMQMessagingClientProperties.getUsername(),
                rabbitMQMessagingClientProperties.getPassword());
    }

    private CredentialsProvider createOAuthCredentialsProvider(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        OAuth2ClientCredentialsGrantCredentialsProvider.OAuth2ClientCredentialsGrantCredentialsProviderBuilder credentialsProviderBuilder = new OAuth2ClientCredentialsGrantCredentialsProvider.OAuth2ClientCredentialsGrantCredentialsProviderBuilder()
                .clientId(rabbitMQMessagingClientProperties.getUsername())
                .clientSecret(rabbitMQMessagingClientProperties.getPassword())
                .tokenEndpointUri(rabbitMQMessagingClientProperties.getKeycloakTokenUrl())
                .grantType(rabbitMQMessagingClientProperties.getGrantType());

        if (rabbitMQMessagingClientProperties.getOauthParameters() != null) {
            rabbitMQMessagingClientProperties.getOauthParameters().forEach(credentialsProviderBuilder::parameter);
        }
        return credentialsProviderBuilder.build();
    }

    @Bean
    public CredentialsRefreshService credentialsRefreshService(RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties) {
        if (StringUtils.isBlank(rabbitMQMessagingClientProperties.getKeycloakTokenUrl())) {
            return null;
        } else {
            Function<Duration, Duration> refreshDelayStrategy =
                    DefaultCredentialsRefreshService.fixedDelayBeforeExpirationRefreshDelayStrategy(Duration.ofSeconds(20));
            return new DefaultCredentialsRefreshService.DefaultCredentialsRefreshServiceBuilder()
                    .refreshDelayStrategy(refreshDelayStrategy)
                    .build();
        }
    }

}