package com.finconsgroup.itserr.marketplace.usercommunication.dm.component;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.User;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.WebSocketAuthentication;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.WebSocketUser;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.OfflineMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.SessionManagementService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.event.EventListener;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.messaging.simp.stomp.StompHeaderAccessor;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.messaging.SessionConnectedEvent;
import org.springframework.web.socket.messaging.SessionDisconnectEvent;
import org.springframework.web.socket.messaging.SessionSubscribeEvent;

import java.time.LocalDateTime;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Listens to WebSocket connect/subscribe/disconnect events to update user presence
 * and trigger offline message delivery and notifications.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class WebSocketEventListener implements InitializingBean, DisposableBean {

    private final SessionManagementService sessionManagementService;
    private final SimpMessagingTemplate messagingTemplate;
    private final OfflineMessageService offlineMessageService;
    private final MessageDestinations messageDestinations;

    private ScheduledExecutorService scheduledExecutor;

    @EventListener
    public void handleWebSocketConnectListener(SessionConnectedEvent event) {
        log.info("WebSocket connection established: {}", event.getMessage());
        StompHeaderAccessor headerAccessor = StompHeaderAccessor.wrap(event.getMessage());
        String sessionId = headerAccessor.getSessionId();

        // Extract user information from session attributes or headers
        WebSocketUser user = extractUserFromHeaders(headerAccessor);

        if (user != null && sessionId != null) {
            User updatedUser = sessionManagementService.onSessionCreated(sessionId, user);
            if (updatedUser != null) {
                broadcastUserStatusChange(updatedUser);
            }
        }
    }

    @EventListener
    public void handleWebSocketDisconnectListener(SessionDisconnectEvent event) {
        StompHeaderAccessor headerAccessor = StompHeaderAccessor.wrap(event.getMessage());
        String sessionId = headerAccessor.getSessionId();

        if (sessionId != null) {
            User updatedUser = sessionManagementService.onSessionDestroyed(sessionId);
            if (updatedUser != null) {
                broadcastUserStatusChange(updatedUser);
            }
        }
    }

    @EventListener
    public void handleWebSocketSubscribeListener(SessionSubscribeEvent event) {
        StompHeaderAccessor headerAccessor = StompHeaderAccessor.wrap(event.getMessage());
        String sessionId = headerAccessor.getSessionId();
        String destination = headerAccessor.getDestination();

        // Extract user information from session attributes or headers
        WebSocketUser user = extractUserFromHeaders(headerAccessor);

        if (user != null && sessionId != null) {
            UUID userId = user.getUserId();
            int userSessionCount = sessionManagementService.getUserSessionCount(userId);

            // Handle offline messages and notifications when subscribing to message queues
            // This ensures the client is ready to receive the messages
            if (destination != null && messageDestinations.isUserMessagesOrTopicConversation(destination)) {
                if (userSessionCount == 1) {
                    // Small delay to ensure subscription is fully established
                    this.scheduledExecutor.schedule(() -> {
                        try {
                            // Check if user has offline messages
                            long messageCount = offlineMessageService.getUndeliveredMessageCount(userId);
                            if (messageCount > 0) {
                                // Deliver offline messages (this will also send notification)
                                offlineMessageService.deliverOfflineMessages(userId);
                            } else {
                                // No offline messages, but still notify if user was offline
                                // This covers the case where user reconnects but has no pending messages
                                log.debug("User {} came online with no pending messages", userId);
                            }
                        } catch (Exception e) {
                            log.warn("Error while handling offline messages for user {}", userId);
                        }
                    }, 200, TimeUnit.MILLISECONDS);
                }
            }

            // Send notification about pending messages when subscribing to notifications queue
            if (destination != null && messageDestinations.isUserNotifications(destination)) {
                // Always send notification when subscribing to notifications queue, regardless of session count
                // This ensures first-time connections get notifications
                this.scheduledExecutor.schedule(() -> {
                    try {
                        // Use notifyUserAboutPendingMessages instead of sendOfflineMessageNotification
                        // This method includes sender information and tells user they need to connect
                        offlineMessageService.notifyUserAboutPendingMessages(userId);
                        log.info("Sent detailed notification with sender info for user {} after subscription to notifications queue", userId);
                    } catch (Exception e) {
                        log.warn("Error while sending pending messages notification to user {}", userId);
                    }
                }, 500, TimeUnit.MILLISECONDS);
            }

            log.info("User {} subscribed to {} (session: {}, total sessions: {})",
                    userId, destination, sessionId, userSessionCount);
        }
    }

    @Override
    public void afterPropertiesSet() {
        log.info("Started the web socket event listener scheduled executor");
        this.scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
    }

    @Override
    public void destroy() {
        if (this.scheduledExecutor != null) {
            this.scheduledExecutor.shutdown();
            log.info("Stopped the web socket event listener scheduled executor");
        }
    }

    private WebSocketUser extractUserFromHeaders(StompHeaderAccessor headerAccessor) {
        // Try to get user ID from session attributes first
        if (headerAccessor.getUser() instanceof WebSocketAuthentication authentication) {
            return authentication.getPrincipal();
        }

        return null;
    }

    private void broadcastUserStatusChange(User user) {
        try {
            // Create status update message
            Map<String, Object> statusUpdate = new LinkedHashMap<>(Map.of(
                    "type", "USER_STATUS_UPDATE",
                    "userId", user.getId(),
                    "online", user.isOnline(),
                    "timestamp", LocalDateTime.now().toString(),
                    "lastSeen", user.getLastSeen()
            ));

            // Broadcast to all a pub-sub user-status topic
            // This could be optimized by maintaining a user-to-conversations mapping
            messagingTemplate.convertAndSend(messageDestinations.getTopicUserStatus(), statusUpdate);

            log.debug("Broadcasted status change for user {}: online={}", user.getId(), user.isOnline());
        } catch (Exception e) {
            log.error("Failed to broadcast status change for user {}: {}", user.getId(), e.getMessage(), e);
        }
    }
}