package com.finconsgroup.itserr.marketplace.usercommunication.dm.component;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.client.dto.userprofile.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.client.dto.userprofile.OutputUserProfileDmDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.User;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.ConversationMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.UserMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.UserRepository;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;

/**
 * Helper class for some common functions related to Conversations.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class ConversationHelper {
    private final ConversationRepository conversationRepository;
    private final ConversationMapper conversationMapper;
    private final UserRepository userRepository;
    private final UserMapper userMapper;
    private final UserProfileDmClient userProfileDmClient;

    /**
     * Saves the users from the conversation in the user table.
     * This will allow to search for conversations that contain name of the users.
     *
     * @param userIds the ids of the user to save.
     */
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void saveConversationUsers(Set<UUID> userIds) {
        Map<UUID, OutputUserProfileDmDto> userProfileDtoById = loadUserProfiles(userIds);
        Map<UUID, User> userById = loadChatUsers(userIds, Function.identity());
        userProfileDtoById.values().forEach(userProfileDmDto -> {
            try {
                User existingUser = userById.get(userProfileDmDto.getId());
                if (existingUser != null) {
                    existingUser.setFirstName(userProfileDmDto.getFirstName());
                    existingUser.setLastName(userProfileDmDto.getLastName());
                    userRepository.saveAndFlush(existingUser);
                } else {
                    User newUser = userMapper.userProfileDmDtoToEntity(userProfileDmDto);
                    newUser.setLastSeen(null);
                    userRepository.saveAndFlush(newUser);
                }
            } catch (Exception e) {
                log.error("Failed to save conversation user: {}", userProfileDmDto.getId(), e);
            }
        });
    }

    /**
     * Load conversations for the provided ids.
     * Also populate the name and receiver for direct conversations from the other user.
     *
     * @param userId          the user id
     * @param conversationIds the conversation ids to load
     * @return the list of {@link OutputConversationDto}
     */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @NonNull
    @SuppressWarnings("squid:S6809")
    public List<OutputConversationDto> loadConversations(@NonNull UUID userId, @NonNull Set<UUID> conversationIds) {
        List<Conversation> conversations = conversationRepository.findAllById(conversationIds);
        return populateDetailsForDirect(userId, conversations);
    }

    /**
     * Populate the name and receiver for direct conversations from the other user.
     *
     * @param userId        the user id
     * @param conversations the conversations to map from
     * @return the list of {@link OutputConversationDto}
     */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @NonNull
    @SuppressWarnings("squid:S6809")
    public List<OutputConversationDto> populateDetailsForDirect(@NonNull UUID userId,
                                                                @NonNull List<Conversation> conversations) {
        List<OutputConversationDto> conversationDtos = new LinkedList<>();
        Set<UUID> directConversationOtherUserIds = new LinkedHashSet<>();
        directConversationOtherUserIds.add(userId);
        conversations
                .forEach(conversation -> {
                    OutputConversationDto conversationDto = conversationMapper.entityToOutputConversationDto(conversation);
                    conversationDtos.add(conversationDto);

                    if (conversation.isDirect()) {
                        UUID otherUserId = conversation.getOtherParticipantId(userId);
                        // temporarily set name to other user id, which will be resolved with actual name later
                        conversationDto.setName(otherUserId.toString());
                        directConversationOtherUserIds.add(otherUserId);
                    }

                });
        populateDetailsForDirect(conversationDtos, directConversationOtherUserIds);
        return conversationDtos;
    }

    /**
     * Populate the name and receiver for direct conversation from the other user.
     *
     * @param userId       the user id
     * @param conversation the conversation to map from
     * @return the  {@link OutputConversationDto}
     */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @NonNull
    public OutputConversationDto populateDetailsForDirect(@NonNull UUID userId, @NonNull Conversation conversation) {
        @SuppressWarnings("squid:S6809")
        List<OutputConversationDto> conversationDtos = populateDetailsForDirect(userId, List.of(conversation));
        return conversationDtos.getFirst();
    }

    /**
     * Populate the name and receiver for direct conversations from the other user.
     *
     * @param conversationDtos          the conversationDtos to update
     * @param directConversationUserIds the user ids of conversation that are of DIRECT type
     */
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public void populateDetailsForDirect(@NonNull List<OutputConversationDto> conversationDtos,
                                         @NonNull Set<UUID> directConversationUserIds) {
        if (directConversationUserIds.isEmpty()) {
            return;
        }

        Map<UUID, OutputUserDto> userDtoById = mapUserIdsToDtos(directConversationUserIds, true);

        conversationDtos.forEach(conversationDto -> {
            UUID receiverId = UUID.fromString(conversationDto.getName());
            OutputUserDto userDto = userDtoById.get(receiverId);
            if (userDto != null) {
                // fallback use name from local user repo
                conversationDto.setName(userDto.getFirstName() + " " + userDto.getLastName());
            }
            conversationDto.setReceiver(userDto);

            conversationDto.setCreatedByUser(userDtoById.get(conversationDto.getCreatedBy()));
        });
    }

    /**
     * Maps the user ids to the dtos populating details like imageUrl etc.
     *
     * @param userIds the user ids
     * @return the map containing the user dto by user id
     */
    @NonNull
    public Map<UUID, OutputUserDto> mapUserIdsToDtos(@NonNull Set<UUID> userIds, boolean loadUserProfiles) {
        if (userIds.isEmpty()) {
            return Map.of();
        }

        Map<UUID, OutputUserProfileDmDto> userProfileDtoById = loadUserProfiles ? loadUserProfiles(userIds) : Map.of();
        Map<UUID, OutputUserDto> userDtoById = loadChatUsers(userIds);

        for (UUID userId : userIds) {
            OutputUserProfileDmDto userProfileDto = userProfileDtoById.get(userId);
            OutputUserDto userDto = userDtoById.get(userId);
            if (userProfileDto != null) {
                // if the user never connected to chat service, then create the user object from user profile response
                if (userDto == null) {
                    userDto = userMapper.userProfileDmDtoToOutputUserDto(userProfileDto);
                    userDtoById.put(userId, userDto);
                } else {
                    userDto.setImageUrl(userProfileDto.getImageUrl());
                }
            }
        }

        return userDtoById;
    }

    @NonNull
    private Map<UUID, OutputUserDto> loadChatUsers(@NonNull Set<UUID> userIds) {
        return loadChatUsers(userIds, userMapper::entityToOutputUserDto);
    }

    @NonNull
    private <T> Map<UUID, T> loadChatUsers(@NonNull Set<UUID> userIds, @NonNull Function<User, T> mapper) {
        Map<UUID, T> mappedUserById = new HashMap<>();
        try {
            userRepository.findAllById(userIds)
                    .forEach(user ->
                            mappedUserById.put(user.getId(), mapper.apply(user))
                    );
        } catch (Exception e) {
            log.error("Error loading chat users", e);
        }
        return mappedUserById;
    }

    @NonNull
    private Map<UUID, OutputUserProfileDmDto> loadUserProfiles(@NonNull Set<UUID> userProfileIds) {
        if (userProfileIds.isEmpty()) {
            return Map.of();
        }

        Map<UUID, OutputUserProfileDmDto> userProfileDmDtoById = new HashMap<>();
        try {
            InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto = InputFindUserProfilesByIdsDto.builder()
                    .ids(new ArrayList<>(userProfileIds))
                    .build();
            OutputPageDto<OutputUserProfileDmDto> userOutputProfileDmPageDto = userProfileDmClient.findAllByIds(
                    inputFindUserProfilesByIdsDto, 0, userProfileIds.size(), "id", SortDirection.ASC);
            userOutputProfileDmPageDto.getContent().forEach(outputUserProfileDmDto ->
                    userProfileDmDtoById.put(outputUserProfileDmDto.getId(), outputUserProfileDmDto));
        } catch (Exception e) {
            log.error("Error loading user profiles", e);
        }
        return userProfileDmDtoById;
    }
}
