package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.DiscussionApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputDiscussionDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputDiscussionDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputDiscussionLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.DiscussionService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link DiscussionApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class DiscussionController implements DiscussionApi {
    private final DiscussionService discussionService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputDiscussionDto upsertDiscussion(String id, InputDiscussionDto dto) {
        return discussionService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputDiscussionDto getDiscussion(String id) {
        return discussionService.getDocument(id);
    }

    @Override
    public void deleteDiscussion(String id) {
        discussionService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        discussionService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return discussionService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputDiscussionLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                               @Nullable String filters,
                                                               int pageNumber,
                                                               int pageSize,
                                                               String sort,
                                                               Sort.Direction direction,
                                                               String multiSort) {

        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return discussionService.getLocalSearch(terms, filters, pageable);
    }
}
