package com.finconsgroup.itserr.marketplace.search.dm.config;

import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.search.dm.bean.QuerySearchFields;
import com.finconsgroup.itserr.marketplace.search.dm.enums.AssociationCategory;
import jakarta.annotation.PostConstruct;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.WeakHashMap;

/**
 * It contains configuration for Global Search
 *
 * @param indexName                 the name of the index or alias
 * @param defaultSortFields         the fields to sort by default
 * @param defaultSortFieldSeparator the separator used to split sort field name and direction
 * @param sortFilterPropertyMap     the map containing mapped field for the sort/filter field e.g. sort/filter cannot be applied to text fields
 *                                  so we need the mapped keyword field
 * @param includeScore              flag to indicate to include score in the response
 * @param termSearchFields          the list of {@link QuerySearchFields} to search for keyword terms
 * @param nestedPaths               the set of nested paths (fields of nested type)
 * @param useWildcard               flag to indicate if wildcard should be used for each term
 * @param enableDeleteAll           flag to indicate if delete all endpoint is enabled
 * @param autoCompletion            the {@link AutoCompletionProperties} to be used for autocompletion queries
 * @param globalSearch              the {@link GlobalSearchProperties} to be used for global search queries
 * @param filter                    {@link FilterProperties} to be used for filter processing
 * @param local                     {@link LocalProperties} to be used for local search processing
 * @param favourite                 {@link FavouriteProperties} to be used for favourite search processing
 */
@Builder
@Slf4j
public record SearchProperties(String indexName,
                               List<String> defaultSortFields,
                               String defaultSortFieldSeparator,
                               Map<String, String> sortFilterPropertyMap,
                               boolean includeScore,
                               List<QuerySearchFields> termSearchFields,
                               Set<String> nestedPaths,
                               boolean useWildcard,
                               boolean enableDeleteAll,
                               AutoCompletionProperties autoCompletion,
                               GlobalSearchProperties globalSearch,
                               FilterProperties filter,
                               LocalProperties local,
                               FavouriteProperties favourite,
                               List<SyncProperties> syncList) {

    // cache to store the computed property type map
    private static final Map<SearchProperties, Map<String, String>> propertyTypeMapCache = new WeakHashMap<>();

    @PostConstruct
    public void init() {
        // trigger calculation of default property type map
        this.defaultPropertyTypeMap();
    }

    /**
     * Returns the default property type map based on the term search fields
     *
     * @return the computed property type map
     */
    @NonNull
    public Map<String, String> defaultPropertyTypeMap() {
        return propertyTypeMapCache.computeIfAbsent(this, sp -> {
            log.info("Calculating default property type map for search properties for index - {}", indexName);
            final Map<String, String> propertyMap = new LinkedHashMap<>();
            Optional.ofNullable(termSearchFields).orElse(List.of())
                    .forEach(qsf -> {
                        qsf.fieldNames().forEach(fn -> propertyMap.put(fn, qsf.fieldType()));
                    });
            return propertyMap;
        });
    }

    /**
     * It contains the configuration for Global Search AutoCompletion
     *
     * @param topHitsLimit the limit on number of records to return for each aggregation term value
     * @param sourceFields the source fields to return in the response for top hits for aggregation
     * @param aggregation  the {@link AggregationProperties} to be used for aggregation (optional)
     */
    @Builder
    public record AutoCompletionProperties(Integer topHitsLimit,
                                           List<String> sourceFields,
                                           AggregationProperties aggregation) {
    }

    /**
     * It contains the configuration for Global Search AutoCompletion
     *
     * @param topHitsLimit the limit on number of records to return for each category
     * @param sourceFields the source fields to return in the response for each category
     * @param aggregation  the {@link AggregationProperties} to be used for aggregation (optional)
     */
    @Builder
    public record GlobalSearchProperties(Integer topHitsLimit,
                                         List<String> sourceFields,
                                         AggregationProperties aggregation) {
    }

    /**
     * It contains the configuration for Aggregation
     *
     * @param term the field on which to perform the aggregation
     * @param name the name of the aggregation
     */
    @Builder
    public record AggregationProperties(String term,
                                        String name) {
    }

    /**
     * It contains the configuration for Local Search
     *
     * @param sourceFields the source fields to return in the response for local search
     */
    @Builder
    public record LocalProperties(List<String> sourceFields) {
    }

    /**
     * It contains the configuration for Favourite Search
     *
     * @param sourceFields the source fields to return in the response for favourite search
     */
    @Builder
    public record FavouriteProperties(List<String> sourceFields) {
    }

    /**
     * It contains the configuration related to document sync between different indexes.
     *
     * @param associationCategory  the category of the associated document
     * @param foreignKeyProperties the fields representing the foreign key of associated document
     * @param fetchPageSize        the page size to fetch documents to sync
     */
    @Builder
    public record SyncProperties(AssociationCategory associationCategory,
                                 List<String> foreignKeyProperties,
                                 int fetchPageSize) {
    }

}
