package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.bean.QueryFilter;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.UserProfileSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputUserProfileLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.event.UserProfileUpdatedEvent;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.UserProfileMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.UserProfileRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.UserProfileService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Default implementation of {@link UserProfileService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultUserProfileService implements UserProfileService {

    private final UserProfileRepository userProfileRepository;
    private final UserProfileMapper userProfileMapper;
    private final UserProfileSearchProperties userProfileSearchProperties;
    private final ApplicationEventPublisher applicationEventPublisher;
    private final Map<String, String> sortFilterPropertyMap;


    public DefaultUserProfileService(UserProfileRepository userProfileRepository,
                                     UserProfileMapper userProfileMapper,
                                     UserProfileSearchProperties userProfileSearchProperties,
                                     DefaultSearchProperties defaultSearchProperties,
                                     ApplicationEventPublisher applicationEventPublisher) {
        this.userProfileRepository = userProfileRepository;
        this.userProfileMapper = userProfileMapper;
        this.userProfileSearchProperties = userProfileSearchProperties;
        this.applicationEventPublisher = applicationEventPublisher;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(userProfileSearchProperties.search(),
                defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputUserProfileDto upsertDocument(@NonNull InputUserProfileDto dto) {
        Optional<UserProfile> existingUserProfileOpt = userProfileRepository.findById(dto.getId());
        UserProfile userProfile = userProfileMapper.toEntity(dto);
        UserProfile savedUserProfile = userProfileRepository.save(userProfile);
        publishEvents(savedUserProfile, existingUserProfileOpt);
        return userProfileMapper.toDto(savedUserProfile);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputUserProfileDto getDocument(@NonNull String id) {
        UserProfile savedUserProfile = userProfileRepository
                .findById(id)
                .filter(UserProfile::isPublicProfile)
                .orElseThrow(() -> new WP2ResourceNotFoundException("search_dm_profile_not_found"));
        return userProfileMapper.toDto(savedUserProfile);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!userProfileRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_profile_not_found");
        }

        userProfileRepository.deleteById(id);
    }

    @Transactional
    @Override
    public void deleteAll() {
        if (!userProfileSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_profile_delete_all_not_enabled");
        }

        userProfileRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                userProfileSearchProperties.search().autoCompletion().sourceFields(),
                userProfileMapper::toAutoCompleteDataDto,
                PageRequest.of(0, userProfileSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                    .builder()
                    .category(Category.PEOPLE.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }

    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputUserProfileLocalSearchDto> getLocalSearch(String terms, String filters, @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(terms)
                .queryFilters(addDefaultFilters(buildQueryFilters(filters, sortFilterPropertyMap)))
                .build();
        Pageable sortedPageable = applySort(pageable, userProfileSearchProperties.search(), sortFilterPropertyMap);
        return search(searchRequest,
                userProfileSearchProperties.search().local().sourceFields(),
                userProfileMapper::toLocalSearchDto,
                sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Page<OutputGlobalSearchDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                userProfileSearchProperties.search().globalSearch().sourceFields(),
                userProfileMapper::toGlobalSearchDataDto,
                PageRequest.of(0, userProfileSearchProperties.search().globalSearch().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchDto
                    .builder()
                    .category(Category.PEOPLE.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }

    private void publishEvents(@NonNull UserProfile savedUserProfile, @NonNull Optional<UserProfile> existingUserProfileOpt) {
        applicationEventPublisher.publishEvent(new UserProfileUpdatedEvent(savedUserProfile, existingUserProfileOpt.orElse(null)));
    }

    @NonNull
    @Override
    public Class<UserProfile> getDocumentClass() {
        return UserProfile.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return userProfileSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return userProfileRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return userProfileRepository;
    }

    private List<QueryFilter> addDefaultFilters(@NonNull List<QueryFilter> queryFilters) {
        List<QueryFilter> filtersWithDefaults = new ArrayList<>(queryFilters);
        filtersWithDefaults.add(QueryFilter.builder()
                .fieldName("publicProfile").filterValues(List.of(Boolean.TRUE.toString()))
                .build());
        return filtersWithDefaults;
    }
}
