package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.CatalogService;
import com.finconsgroup.itserr.marketplace.search.dm.service.DiscussionService;
import com.finconsgroup.itserr.marketplace.search.dm.service.EventService;
import com.finconsgroup.itserr.marketplace.search.dm.service.GlobalSearchService;
import com.finconsgroup.itserr.marketplace.search.dm.service.InstitutionalPageService;
import com.finconsgroup.itserr.marketplace.search.dm.service.NewsService;
import com.finconsgroup.itserr.marketplace.search.dm.service.SearchService;
import com.finconsgroup.itserr.marketplace.search.dm.service.UserProfileService;
import jakarta.annotation.PreDestroy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Function;

/**
 * Default implementation of {@link GlobalSearchService} to perform search operations by connecting to an
 * OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultGlobalSearchService implements GlobalSearchService {

    private final List<SearchService<?, ?>> searchServices;
    private final ExecutorService executor;

    public DefaultGlobalSearchService(CatalogService catalogService,
                                      UserProfileService userProfileService,
                                      InstitutionalPageService institutionalPageService,
                                      EventService eventService,
                                      NewsService newsService,
                                      DiscussionService discussionService) {
        this.searchServices = List.of(
                catalogService, userProfileService, institutionalPageService, eventService, newsService, discussionService
        );
        this.executor = Executors.newVirtualThreadPerTaskExecutor();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        List<Function<String, List<OutputGlobalSearchAutoCompleteDto>>> operations =
                new ArrayList<>(searchServices.size());
        for (SearchService<?, ?> searchService : searchServices) {
            operations.add(searchService::getAutoCompletions);
        }
        return performOperations(operations, terms);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        List<Function<String, List<OutputGlobalSearchDto>>> operations =
                new ArrayList<>(searchServices.size());
        for (SearchService<?, ?> searchService : searchServices) {
            operations.add(searchService::getSearch);
        }
        return performOperations(operations, terms);
    }

    @PreDestroy
    void close() {
        executor.shutdown();
    }

    private <T> List<T> performOperations(List<Function<String, List<T>>> operations, String input) {
        List<T> resultList = new ArrayList<>();

        List<Future<List<T>>> resultFutures = new ArrayList<>(searchServices.size());
        for (Function<String, List<T>> operation : operations) {
            Future<List<T>> resultFuture = executor.submit(() -> operation.apply(input));
            resultFutures.add(resultFuture);
        }

        for (Future<List<T>> resultFuture : resultFutures) {
            try {
                resultList.addAll(resultFuture.get());
            } catch (ExecutionException e) {
                throw new WP2ExecutionException(e.getCause());
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new WP2ExecutionException(e);
            }
        }
        return resultList;
    }
}
