package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.EventApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputEventLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.EventService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link EventApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class EventController implements EventApi {
    private final EventService eventService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputEventDto upsertEvent(String id, InputEventDto dto) {
        return eventService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputEventDto getEvent(String id) {
        return eventService.getDocument(id);
    }

    @Override
    public void deleteEvent(String id) {
        eventService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        eventService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return eventService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputEventLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                          @Nullable String filters,
                                                          int pageNumber,
                                                          int pageSize,
                                                          String sort,
                                                          Sort.Direction direction,
                                                          String multiSort) {

        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return eventService.getLocalSearch(terms, filters, pageable);
    }
}
