package com.finconsgroup.itserr.marketplace.search.dm.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.core.web.validation.QueryFilterValidator;
import com.finconsgroup.itserr.marketplace.search.dm.constant.FilterConfigKeys;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CatalogRepository;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.ssl.SSLContextBuilder;
import org.opensearch.client.RestClientBuilder;
import org.opensearch.client.json.JsonpMapper;
import org.opensearch.client.json.jackson.JacksonJsonpMapper;
import org.opensearch.spring.boot.autoconfigure.RestClientBuilderCustomizer;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.elasticsearch.repository.config.EnableElasticsearchRepositories;
import org.springframework.data.web.config.EnableSpringDataWebSupport;

import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

import static org.springframework.data.web.config.EnableSpringDataWebSupport.PageSerializationMode.VIA_DTO;

/**
 * Configuration for beans to be created/customized for Open Search.
 */
@Configuration
@EnableSpringDataWebSupport(pageSerializationMode = VIA_DTO)
@EnableElasticsearchRepositories(basePackageClasses = CatalogRepository.class,
    repositoryImplementationPostfix = "OpenSearchImpl")
public class SearchConfiguration {

    /**
     * Allow to connect to the OpenSearch instance which uses self-signed certificates
     */
    @Bean
    RestClientBuilderCustomizer customizer(@Value("${search.rest-client.allow-self-ssl:false}") boolean allowSelfSsl) {
        return new RestClientBuilderCustomizer() {
            @Override
            public void customize(HttpAsyncClientBuilder builder) {
                if (allowSelfSsl) {
                    try {
                        builder.setSSLContext(new SSLContextBuilder()
                            .loadTrustMaterial(null, new TrustSelfSignedStrategy())
                            .build());
                    } catch (final KeyManagementException | NoSuchAlgorithmException | KeyStoreException ex) {
                        throw new RuntimeException("Failed to initialize SSL Context instance", ex);
                    }
                }
            }

            @Override
            public void customize(RestClientBuilder builder) {
                // No additional customizations needed
            }
        };
    }

    /**
     * Override the default object mapper
     *
     * @param objectMapper the springs auto configured object mapper
     * @return the @{@link JsonpMapper} instance
     */
    @Bean
    public JsonpMapper jacksonJsonpMapper(ObjectMapper objectMapper) {
        return new JacksonJsonpMapper(objectMapper);
    }

    /**
     * Defines the filter properties map bean needed by the {@link QueryFilterValidator}
     *
     * @param catalogSearchProperties the catalog search configuration
     * @param userProfileSearchProperties the user profile search configuration
     * @return the map filter properties for default key
     */
    @Bean(QueryFilterValidator.FILTER_PROPERTY_MAP_BEAN_NAME)
    public Map<String, FilterProperties> filterPropertyMap(CatalogSearchProperties catalogSearchProperties,
                                                           UserProfileSearchProperties userProfileSearchProperties,
                                                           InstitutionalPageSearchProperties institutionalPageSearchProperties,
                                                           EventSearchProperties eventSearchProperties,
                                                           NewsSearchProperties newsSearchProperties) {
        Map<String, FilterProperties> filterPropertyMap = new HashMap<>();
        if (catalogSearchProperties.search().filter() != null) {
            filterPropertyMap.put(FilterConfigKeys.CATALOG, catalogSearchProperties.search().filter());
        }
        if (userProfileSearchProperties.search().filter() != null) {
            filterPropertyMap.put(FilterConfigKeys.USER_PROFILE, userProfileSearchProperties.search().filter());
        }
        if (institutionalPageSearchProperties.search().filter() != null) {
            filterPropertyMap.put(FilterConfigKeys.INSTITUTIONAL_PAGE, institutionalPageSearchProperties.search().filter());
        }
        if (eventSearchProperties.search().filter() != null) {
            filterPropertyMap.put(FilterConfigKeys.EVENT, eventSearchProperties.search().filter());
        }
        if (newsSearchProperties.search().filter() != null) {
            filterPropertyMap.put(FilterConfigKeys.NEWS, newsSearchProperties.search().filter());
        }
        return Map.copyOf(filterPropertyMap);
    }

}
