package com.finconsgroup.itserr.marketplace.search.dm.opensearch;

import com.finconsgroup.itserr.marketplace.core.web.bean.QueryFilter;
import com.finconsgroup.itserr.marketplace.search.dm.bean.ContainsAnyTermsRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.ContainsAnyTermsWithFiltersRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.QueryRequest;
import com.finconsgroup.itserr.marketplace.search.dm.bean.QuerySearchFields;
import com.finconsgroup.itserr.marketplace.search.dm.bean.TopHitsAggregationRequest;
import com.finconsgroup.itserr.marketplace.search.dm.entity.ScoredDocument;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.opensearch.client.json.JsonData;
import org.opensearch.client.opensearch._types.FieldValue;
import org.opensearch.client.opensearch._types.SortOptions;
import org.opensearch.client.opensearch._types.SortOrder;
import org.opensearch.client.opensearch._types.aggregations.Aggregate;
import org.opensearch.client.opensearch._types.query_dsl.BoolQuery;
import org.opensearch.client.opensearch._types.query_dsl.Query;
import org.opensearch.client.opensearch._types.query_dsl.RegexpQuery;
import org.opensearch.client.opensearch._types.query_dsl.TermQuery;
import org.opensearch.client.opensearch.core.SearchRequest;
import org.opensearch.client.opensearch.core.SearchResponse;
import org.opensearch.client.util.ObjectBuilder;
import org.opensearch.data.core.OpenSearchOperations;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.IndexOperations;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.finconsgroup.itserr.marketplace.core.web.utils.FilterUtils.RANGE_FILTER_OPERATORS;

/**
 * Helper class for OpenSearch related functionality.
 */
@Component
@Slf4j
public class OpenSearchHelper {
    private static final String CONTAINS_WILDCARD_FORMAT = "*%s*^0.5 %s"; // reduce the score for partial match
    private static final String TOP_HITS_AGGREGATION_FORMAT = "top_hits_%s";
    private static final String ID_FIELD_NAME = "_id";
    private static final Set<Character> WILDCARD_SPECIAL_CHARS = Set.of(':', '.', '?', '+', '*', '|', '{', '}', '[',
        ']', '(', ')', '\"', '\\', '@', '&', '~', '<', '>');
    private static final Set<String> CASE_SENSITIVE_TYPES = Set.of("text", "keyword");
    /**
     * Separator string to combine the tokens in open search regexp query for multi terms match
     */
    private static final String REGEXP_SEPARATOR = "|";

    private final OpenSearchOperations openSearchOperations;
    private final boolean indexMetadataOperationsEnabled;
    private final Duration indexMetadataOperationsRetryDelay;
    private final Map<String, PropertyTypeMapState> indexPropertyMapState = new LinkedHashMap<>();

    public OpenSearchHelper(
            OpenSearchOperations openSearchOperations,
            @Value("${search-dm.index-metadata-operations-enabled}") boolean indexMetadataOperationsEnabled,
            @Value("${search-dm.index-metadata-operations-retry-delay}") Duration indexMetadataOperationsRetryDelay) {
        this.openSearchOperations = openSearchOperations;
        this.indexMetadataOperationsEnabled = indexMetadataOperationsEnabled;
        this.indexMetadataOperationsRetryDelay = indexMetadataOperationsRetryDelay;
    }

    /**
     * It creates a search request to aggregate the hits by provided term and return only the top hits for each of the
     * matching term values.
     *
     * @param queryRequest the request to filter query results
     * @param topHitsLimit the number of top hits to return
     * @param indexNames   the names of the indices to search
     * @param <T>          The type of the document to query for
     * @return SearchRequest for the aggregation using the query
     */
    public <T> SearchRequest buildTopHitsSearchRequest(
        QueryRequest<T> queryRequest,
        int topHitsLimit,
        String[] indexNames) {

        Function<Query.Builder, ObjectBuilder<Query>> queryFn = mapQueryRequestToQueryFn(queryRequest);

        return SearchRequest.of(r -> {
            r.index(Arrays.asList(indexNames))
             .query(queryFn);
            if (queryRequest.getSourceFields() != null) {
                r.source(s -> s.filter(sf -> sf.includes(queryRequest.getSourceFields())));
            }
            return r.size(topHitsLimit);
        });
    }

    /**
     * It creates a search request to aggregate the hits by provided term and return only the top hits for each of the
     * matching term values.
     *
     * @param queryRequest       the request to filter query results
     * @param aggregationRequest the request containing details for term aggregation and top hits
     * @param indexNames         the names of the indices to search
     * @param <T>                The type of the document to query for
     * @return SearchRequest for the aggregation using the query
     */
    public <T> SearchRequest buildTopHitsByTermSearchRequest(
        QueryRequest<T> queryRequest,
        TopHitsAggregationRequest<T> aggregationRequest,
        String[] indexNames) {

        Function<Query.Builder, ObjectBuilder<Query>> queryFn = mapQueryRequestToQueryFn(queryRequest);

        return SearchRequest.of(r ->
            r.index(Arrays.asList(indexNames))
             .query(queryFn)
             .aggregations(aggregationRequest.name(), a ->
                 a.terms(t -> t.field(aggregationRequest.term()))
                  .aggregations(TOP_HITS_AGGREGATION_FORMAT.formatted(aggregationRequest.name()), thq ->
                      thq.topHits(tha ->
                          tha.size(aggregationRequest.topHitsLimit())
                             .source(s -> s.filter(sf -> sf.includes(aggregationRequest.sourceFields()))))
                  ))
             .size(0) // no need to return actual query results
        );
    }

    /**
     * It creates a search request to find page of documents based on the query and page request
     *
     * @param queryRequest the request to filter query results
     * @param pageable     the page to return
     * @param indexNames   the names of the indices to search
     * @param <T>          The type of the document to query for
     * @return SearchRequest for the page using the query
     */
    public <T> SearchRequest buildFindPageSearchRequest(
        QueryRequest<T> queryRequest,
        Pageable pageable,
        String[] indexNames) {

        Function<Query.Builder, ObjectBuilder<Query>> queryFn = mapQueryRequestToQueryFn(queryRequest);

        return SearchRequest.of(r -> {
            r.index(Arrays.asList(indexNames))
             .query(queryFn);
            if (queryRequest.getSourceFields() != null) {
                r.source(s -> s.filter(sf -> sf.includes(queryRequest.getSourceFields())));
            }
            if (pageable.getSort().isSorted()) {
                List<SortOptions> sortOptionsList = pageable.getSort().stream().map(order ->
                    SortOptions.of(s -> s.field(f ->
                        f.field(order.getProperty())
                         .order(order.getDirection().isAscending() ? SortOrder.Asc : SortOrder.Desc)))
                ).toList();
                r.sort(sortOptionsList);
            }
            return r.from(pageable.getPageSize() * pageable.getPageNumber()).size(pageable.getPageSize());
        });
    }

    /**
     * It maps the response from the top hits search request to group the list of documents
     * by the term.
     *
     * @param searchResponse the search response to map
     * @param request        the query request
     * @param <T>            the fieldType of the document
     * @return {@code List<T>} contain the list of documents
     */
    public <T> List<T> mapTopHitsSearchResponse(SearchResponse<T> searchResponse, QueryRequest<T> request) {
        return searchResponse.hits().hits().stream().map(h -> {
            T hitResult = h.source();
            if (request.includeScore() && hitResult instanceof ScoredDocument scoredDocument) {
                scoredDocument.setScore(h.score());
            }
            return hitResult;
        }).toList();
    }

    /**
     * It maps the response from the top hits by term aggregation search request to group the list of documents
     * by the term.
     *
     * @param searchResponse the search response to map
     * @param request        the aggregation request
     * @param <T>            the fieldType of the document
     * @return {@code Map<String, T>} contain the list of documents grouped by term
     */
    public <T> Map<String, List<T>> mapTopHitsByTermAggregateResponse(SearchResponse<T> searchResponse, TopHitsAggregationRequest<T> request) {
        Map<String, List<T>> topHitsByTerm = new LinkedHashMap<>();
        Aggregate termsAggregate = searchResponse.aggregations().get(request.name());
        if (termsAggregate != null) {
            termsAggregate.sterms().buckets().array().forEach(b -> {
                Aggregate topHitsAggregate = b.aggregations().get(TOP_HITS_AGGREGATION_FORMAT.formatted(request.name()));
                if (topHitsAggregate != null) {
                    List<T> topHits = new LinkedList<>();
                    topHitsAggregate.topHits().hits().hits().forEach(d -> {
                        T topHitResult = Objects.requireNonNull(d.source()).to(request.documentClass());
                        if (request.includeScore() && topHitResult instanceof ScoredDocument scoredDocument) {
                            scoredDocument.setScore(d.score());
                        }
                        topHits.add(topHitResult);
                    });
                    topHitsByTerm.put(b.key(), topHits);
                }
            });
        }
        return topHitsByTerm;
    }

    /**
     * It maps the response from the page search request to the page of documents.
     *
     * @param searchResponse the search response to map
     * @param request        the query request
     * @param pageable       the page to return
     * @param <T>            the fieldType of the document
     * @return {@code Page<T>} containing page of documents
     */
    public <T> Page<T> mapPageSearchResponse(SearchResponse<T> searchResponse, QueryRequest<T> request, Pageable pageable) {

        List<T> content = searchResponse.hits().hits().stream().map(h -> {
            T hitResult = h.source();
            if (request.includeScore() && hitResult instanceof ScoredDocument scoredDocument) {
                scoredDocument.setScore(h.score());
            }
            return hitResult;
        }).toList();

        return new PageImpl<>(content, pageable, searchResponse.hits().total().value());
    }

    /**
     * It creates a function to build the query for containing any of the terms provided using wildcards.
     *
     * @param request the request to search for
     * @param <T>     the type of the document returned by query
     * @return Function to build the query
     */
    public <T> Function<Query.Builder, ObjectBuilder<Query>> mapContainsAnyTermsRequestToQueryFn(ContainsAnyTermsRequest<T> request) {
        return q ->
            q.bool(b -> {
                    applyContainsAnyTermsRequest(request, b, buildNestedPathByPrefix(request.nestedPaths()));
                    return b;
                }
            );
    }

    /**
     * It creates a function to build the query for containing any of the terms provided using wildcards.
     *
     * @param request the request to search for
     * @param <T>     the type of the document returned by query
     * @return Function to build the query
     */
    public <T> Function<Query.Builder, ObjectBuilder<Query>> mapContainsAnyTermsWithFiltersRequestToQueryFn(ContainsAnyTermsWithFiltersRequest<T> request) {
        final Map<String, String> propertyTypeMap;
        if (request.queryFilters() != null && !request.queryFilters().isEmpty()) {
            // only query metadata when actually needed i.e. when filters are provided
            propertyTypeMap = getIndexPropertyTypeMap(request.getDocumentClass(), request.defaultPropertyTypeMap());
        } else {
            propertyTypeMap = Map.of();
        }
        final Map<String, String> nestedPathByPrefix = buildNestedPathByPrefix(request.nestedPaths());
        return q ->
            q.bool(b -> {
                    if (request.containsAnyTermsRequest() != null) {
                        applyContainsAnyTermsRequest(request.containsAnyTermsRequest(), b, nestedPathByPrefix);
                    }
                    // add ids to must query, if provided
                    if (request.ids() != null && !request.ids().isEmpty()) {
                        List<FieldValue> fieldValues = request.ids().stream()
                                                              .map(String::trim)
                                                              .filter(StringUtils::isNotBlank)
                                                              .map(FieldValue::of)
                                                              .toList();
                        b.must(m -> m.terms(t -> t.field(ID_FIELD_NAME).terms(tf -> tf.value(fieldValues))));
                    }
                    applyQueryFilters(request.queryFilters(), b, nestedPathByPrefix, propertyTypeMap);
                    return b;
                }
            );
    }

    /**
     * Processes the mappings properties to build the property type map.
     *
     * @param path            the current property path
     * @param mappings        the mappings to process
     * @param propertyTypeMap the property type map to update
     */
    private void processMappingProperties(String path, Map<String, Object> mappings, Map<String, String> propertyTypeMap) {
        mappings.forEach((key, value) -> {
            if ("type".equals(key) && value instanceof String strValue) {
                propertyTypeMap.put(path, strValue);
            } else if (value instanceof Map) {
                String newPath;
                if ("properties".equals(key) || "fields".equals(key)) {
                    newPath = path;
                } else if (path == null) {
                    newPath = key;
                } else {
                    newPath = path + "." + key;
                }
                //noinspection unchecked
                processMappingProperties(newPath, (Map<String, Object>) value, propertyTypeMap);
            }
        });
    }

    /**
     * Returns the property type map for the index if fetched already.
     * Otherwise, builds the same and then returns it.
     *
     * @param documentClass the index document class
     * @param defaultPropertyTypeMap  the default property type map
     * @return the property type map
     */
    private Map<String, String> getIndexPropertyTypeMap(Class<?> documentClass, Map<String, String> defaultPropertyTypeMap) {
        String indexKey = documentClass.getSimpleName();
        try {
            Map<String, String> propertyTypeMap = Map.of();
            if (!indexMetadataOperationsEnabled) {
                propertyTypeMap = defaultPropertyTypeMap;
            } else {
                synchronized (indexPropertyMapState) {
                    boolean fetchMappings = false;
                    if (indexPropertyMapState.containsKey(indexKey)) {
                        PropertyTypeMapState propertyTypeMapState = indexPropertyMapState.get(indexKey);
                        if (propertyTypeMapState.isSuccess()) {
                            propertyTypeMap = propertyTypeMapState.getPropertyTypeMap();
                        } else if (!canRetryMetadataOperation(propertyTypeMapState.getLastFailedAt())) {
                            propertyTypeMap = defaultPropertyTypeMap;
                        } else {
                            fetchMappings = true;
                        }
                    } else {
                        fetchMappings = true;
                    }

                    if (fetchMappings) {
                        IndexOperations indexOperations = openSearchOperations.indexOps(documentClass);
                        Map<String, Object> mappings = indexOperations.getMapping();
                        Map<String, String> tempPropertyTypeMap = new LinkedHashMap<>();
                        processMappingProperties(null, mappings, tempPropertyTypeMap);
                        propertyTypeMap = Map.copyOf(tempPropertyTypeMap);
                        // store in cache
                        PropertyTypeMapState propertyTypeMapState = PropertyTypeMapState.success(propertyTypeMap);
                        indexPropertyMapState.put(indexKey, propertyTypeMapState);
                    }
                }
            }
            return propertyTypeMap;
        } catch (Exception e) {
            log.error("Error while getting index property type map for {}", documentClass.getSimpleName(), e);
            indexPropertyMapState.put(indexKey, PropertyTypeMapState.failure(Instant.now()));
            return defaultPropertyTypeMap;
        }
    }

    private boolean canRetryMetadataOperation(Instant lastFailedAt) {
        // wait for retry delay before retrying again
        return Duration.between(lastFailedAt, Instant.now()).compareTo(indexMetadataOperationsRetryDelay) > 0;
    }

    private <T> void applyContainsAnyTermsRequest(@NonNull ContainsAnyTermsRequest<T> request,
                                                  @NonNull BoolQuery.Builder b,
                                                  @NonNull Map<String, String> nestedPathByPrefix) {
        List<String> termsList = Stream.of(request.terms().trim().split(" "))
                                       .map(String::trim)
                                       .filter(s -> !s.isEmpty())
                                       .toList();

        // build the query string for full text search
        final String fullTextSearchValue;
        if (request.useWildcard()) {
            fullTextSearchValue = termsList
                .stream()
                .map(s -> {
                    if (request.useWildcard()) {
                        String escapedStr = escapeWildcardSpecialChars(s);
                        return CONTAINS_WILDCARD_FORMAT.formatted(escapedStr, s);
                    } else {
                        return s;
                    }
                }).collect(Collectors.joining(" "));
        } else {
            fullTextSearchValue = request.terms();
        }

        // build the regexp value for exact search that includes the individual tokens as well as entire term
        final String regexpValue = buildRegexpValue(Stream.concat(Stream.of(request.terms()), termsList.stream()));

        request.searchFields().forEach(sf -> {
            if (QuerySearchFields.TYPE_TEXT.equals(sf.fieldType())) {
                Map<String, List<String>> fieldNamesByNestedPath = findNestedFields(sf.fieldNames(), nestedPathByPrefix);
                if (fieldNamesByNestedPath.size() > 1) {
                    // if there are nested paths, then form a bool query with should for each nested path
                    // and minimum should match = 1
                    b.should(m -> m.bool(fb -> {
                                fieldNamesByNestedPath.forEach((nestedPath, fieldNames) ->
                                        fb.should(fbq ->
                                                applyNestedPath(fbq, nestedPath, qb ->
                                                        qb.queryString(qs ->
                                                                qs.allowLeadingWildcard(request.useWildcard())
                                                                        .fields(fieldNames)
                                                                        .query(fullTextSearchValue)))
                                        ).minimumShouldMatch("1")
                                );
                                return fb;
                            })
                    );
                } else {
                    // no nested paths, so form a single queryString query
                    b.should(m -> {
                        fieldNamesByNestedPath
                                .forEach((nestedPath, fieldNames) ->
                                        applyNestedPath(m, nestedPath, qb ->
                                                qb.queryString(qs ->
                                                        qs.allowLeadingWildcard(request.useWildcard())
                                                                .fields(fieldNames)
                                                                .query(fullTextSearchValue))));
                        return m;
                    });
                }
            } else if (QuerySearchFields.TYPE_KEYWORD.equals(sf.fieldType())) {
                sf.fieldNames().forEach(fn ->
                        b.should(m -> applyNestedPath(m, findNestedPath(fn, nestedPathByPrefix), qb -> qb.regexp(t ->
                                t.field(fn).value(regexpValue).caseInsensitive(true)
                        )))
                );
            }
        });
        b.minimumShouldMatch("1");
    }

    /*
     * Builds a regular expression to match any of the provided terms.
     *
     * E.g. For the terms "Symbols Identity"
     * it will return "Symbols|Identity"
     * which represents a regular expression to match either Symbols or Identity
     */
    private String buildRegexpValue(Stream<String> terms) {
        return terms.map(this::escapeWildcardSpecialChars).collect(Collectors.joining(REGEXP_SEPARATOR));
    }

    private Map<String, String> buildNestedPathByPrefix(Set<String> nestedPaths) {
        if (nestedPaths == null || nestedPaths.isEmpty()) {
            return Map.of();
        }

        return nestedPaths.stream().collect(Collectors.toMap(nestedPath -> nestedPath + ".", Function.identity()));
    }

    private Map<String, List<String>> findNestedFields(List<String> fieldNames, Map<String, String> nestedPathByPrefix) {
        if (nestedPathByPrefix == null || nestedPathByPrefix.isEmpty()) {
            return Map.of("", fieldNames);
        }

        Map<String, List<String>> fieldNamesByNestedPath = new LinkedHashMap<>();
        for (String fieldName : fieldNames) {
            String key = findNestedPath(fieldName, nestedPathByPrefix);
            fieldNamesByNestedPath.computeIfAbsent(key, k -> new LinkedList<>()).add(fieldName);
        }
        return fieldNamesByNestedPath;
    }

    private String findNestedPath(String fieldName, Map<String, String> nestedPathByPrefix) {
        if (nestedPathByPrefix == null || nestedPathByPrefix.isEmpty()) {
            return "";
        }

        return nestedPathByPrefix.keySet().stream().filter(fieldName::startsWith).findFirst().map(nestedPathByPrefix::get).orElse("");
    }

    private void applyQueryFilters(@Nullable List<QueryFilter> queryFilters,
                                   @NonNull BoolQuery.Builder b,
                                   Map<String, String> nestedPathByPrefix,
                                   Map<String, String> propertyTypeMap) {
        if (queryFilters == null || queryFilters.isEmpty()) {
            return;
        }

        // combine all range filters for single field
        // e.g. > and < or >= and <=, so that the range filters can be applied together
        Map<String, List<QueryFilter>> rangeFiltersByField = queryFilters.stream()
                .filter(qf -> RANGE_FILTER_OPERATORS.contains(qf.operator()))
                .collect(Collectors.groupingBy(QueryFilter::fieldName));

        queryFilters.stream()
                .filter(qf -> !RANGE_FILTER_OPERATORS.contains(qf.operator())) // exclude range filters
                .forEach(qf ->
                        b.filter(f ->
                                applyEQFilter(qf, f, nestedPathByPrefix, propertyTypeMap)
                        ));

        rangeFiltersByField.forEach((fieldName, rangeFilters) ->
            b.filter(f -> applyRangeFilters(fieldName, f, nestedPathByPrefix, rangeFilters)
        ));
    }

    private Query.Builder applyEQFilter(QueryFilter qf,
                                        Query.Builder f,
                                        Map<String, String> nestedPathByPrefix,
                                        Map<String, String> propertyTypeMap) {
        if (qf.filterValues().size() > 1) {
            String regexpValue = buildRegexpValue(qf.filterValues().stream());
            applyNestedPath(f, findNestedPath(qf.fieldName(), nestedPathByPrefix), qb ->
                    qb.regexp(t ->
                            applyRegexpCaseInsensitiveFilter(t.field(qf.fieldName()).value(regexpValue),
                                    qf, propertyTypeMap)
                    ));

        } else {
            applyNestedPath(f, findNestedPath(qf.fieldName(), nestedPathByPrefix), qb ->
                    qb.term(t ->
                            applyTermCaseInsensitiveFilter(
                                    t.field(qf.fieldName()).value(FieldValue.of(qf.filterValues().getFirst())),
                                    qf, propertyTypeMap)
                    ));
        }
        return f;
    }

    private Query.Builder applyRangeFilters(String fieldName,
                                            Query.Builder f,
                                            Map<String, String> nestedPathByPrefix,
                                            List<QueryFilter> rangeFilters) {
        applyNestedPath(f, findNestedPath(fieldName, nestedPathByPrefix), qb ->
                qb.range(r -> {
                            r.field(fieldName);

                            rangeFilters.forEach(rf -> {
                                JsonData value = JsonData.of(rf.filterValues().getFirst());
                                switch (rf.operator()) {
                                    case GT -> r.gt(value);
                                    case GTE -> r.gte(value);
                                    case LT -> r.lt(value);
                                    case LTE -> r.lte(value);
                                }
                            });

                            return r;
                        }
                ));
        return f;
    }

    private TermQuery.Builder applyTermCaseInsensitiveFilter(TermQuery.Builder termQueryBuilder,
                                                             QueryFilter queryFilter,
                                                             Map<String, String> propertyTypeMap) {
        if (propertyTypeMap == null) {
            return termQueryBuilder;
        }

        String type = propertyTypeMap.get(queryFilter.fieldName());
        if (type != null && CASE_SENSITIVE_TYPES.contains(type)) {
            return termQueryBuilder.caseInsensitive(Boolean.TRUE);
        } else {
            return termQueryBuilder;
        }
    }

    private RegexpQuery.Builder applyRegexpCaseInsensitiveFilter(RegexpQuery.Builder regexpQueryBuilder,
                                                                 QueryFilter queryFilter,
                                                                 Map<String, String> propertyTypeMap) {
        if (propertyTypeMap == null) {
            return regexpQueryBuilder;
        }

        String type = propertyTypeMap.get(queryFilter.fieldName());
        if (type != null && CASE_SENSITIVE_TYPES.contains(type)) {
            return regexpQueryBuilder.caseInsensitive(Boolean.TRUE);
        } else {
            return regexpQueryBuilder;
        }
    }

    private ObjectBuilder<Query> applyNestedPath(@NonNull Query.Builder queryBuilder,
                                                 String nestedPath,
                                                 @NonNull Function<Query.Builder, ObjectBuilder<Query>> fn) {
        if (StringUtils.isNotBlank(nestedPath)) {
            return queryBuilder.nested(n -> n.path(nestedPath).query(fn));
        } else {
            return fn.apply(queryBuilder);
        }
    }


    // maps the query request to
    public <T> Function<Query.Builder, ObjectBuilder<Query>> mapQueryRequestToQueryFn(QueryRequest<T> request) {
        if (request instanceof ContainsAnyTermsRequest<T> containsAnyTermsRequest) {
            return mapContainsAnyTermsRequestToQueryFn(containsAnyTermsRequest);
        } else if (request instanceof ContainsAnyTermsWithFiltersRequest<T> containsAnyTermsWithFiltersRequest) {
            return mapContainsAnyTermsWithFiltersRequestToQueryFn(containsAnyTermsWithFiltersRequest);
        }

        throw new IllegalArgumentException("Unknown OpenSearchQueryRequest");
    }

    /*
     * Escapes the special characters for open search regex queries with backslash.
     * Not listed, but additionally ':' also needs to be escaped.
     *
     * See link for more details - https://docs.opensearch.org/docs/latest/query-dsl/regex-syntax/
     */
    private String escapeWildcardSpecialChars(String s) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (WILDCARD_SPECIAL_CHARS.contains(c)) {
                sb.append('\\');
            }
            sb.append(c);
        }
        return sb.toString();
    }

    @Data
    @AllArgsConstructor
    private static class PropertyTypeMapState {
        private boolean success;
        private Map<String, String> propertyTypeMap;
        private Instant lastFailedAt;

        private static PropertyTypeMapState success(Map<String, String> propertyMap) {
            return new PropertyTypeMapState(true, propertyMap, null);
        }

        private static PropertyTypeMapState failure(Instant lastFailedAt) {
            return new PropertyTypeMapState(false, null, lastFailedAt);
        }
    }
}
