package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.bean.QueryFilter;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.bean.PostProcessFilterResult;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.DiscussionSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputDiscussionDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputDiscussionDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputDiscussionLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.Discussion;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.event.UserProfileUpdatedEvent;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.DiscussionMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.DiscussionRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.DiscussionService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

/**
 * Default implementation of {@link DiscussionService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultDiscussionService implements DiscussionService, ApplicationListener<UserProfileUpdatedEvent> {

    private static final String CREATOR_USER_ID_FIELD_NAME = "userInfoDTO.id";
    /*
     * Filter key to for visibility, which should be added as an implicit filter.
     */
    private static final String FILTER_KEY_VISIBILITY = "visibility";
    /*
     * Filter values for visibility, by default only public posts should be shown.
     */
    private static final List<String> FILTER_VISIBILITY_DEFAULT_VALUE = List.of("public");

    private final DiscussionRepository discussionRepository;
    private final DiscussionMapper discussionMapper;
    private final DiscussionSearchProperties discussionSearchProperties;
    private final Map<String, String> sortFilterPropertyMap;

    public DefaultDiscussionService(DiscussionRepository discussionRepository,
                                    DiscussionMapper discussionMapper,
                                    DiscussionSearchProperties discussionSearchProperties,
                                    DefaultSearchProperties defaultSearchProperties) {
        this.discussionRepository = discussionRepository;
        this.discussionMapper = discussionMapper;
        this.discussionSearchProperties = discussionSearchProperties;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(discussionSearchProperties.search(),
                defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputDiscussionDto upsertDocument(@NonNull InputDiscussionDto dto) {
        Discussion discussion = discussionMapper.toEntity(dto);
        Discussion savedDiscussion = discussionRepository.save(discussion);
        return discussionMapper.toDto(savedDiscussion);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputDiscussionDto getDocument(@NonNull String id) {
        Discussion savedDiscussion = discussionRepository
                .findById(id)
                .orElseThrow(() -> new WP2ResourceNotFoundException(
                        "search_dm_discussion_not_found"));
        return discussionMapper.toDto(savedDiscussion);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!discussionRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_discussion_not_found");
        }

        discussionRepository.deleteById(id);
    }

    @Override
    public void deleteAll() {
        if (!discussionSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_discussion_delete_all_not_enabled");
        }

        discussionRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                discussionSearchProperties.search().autoCompletion().sourceFields(),
                discussionMapper::toAutoCompleteDataDto,
                PageRequest.of(0, discussionSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                    .builder()
                    .category(Category.DISCUSSION.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }


    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputDiscussionLocalSearchDto> getLocalSearch(String terms, String filters,
                                                               @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(terms)
                .queryFilters(addDefaultFilters(buildQueryFilters(filters, sortFilterPropertyMap)))
                .build();
        Pageable sortedPageable = applySort(pageable, discussionSearchProperties.search(),
                sortFilterPropertyMap);
        return search(searchRequest,
                discussionSearchProperties.search().local().sourceFields(),
                discussionMapper::toLocalSearchDto,
                sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Map<String, List<OutputGlobalSearchDataDto>> searchResultAggregatedByType = searchByAggregation(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                discussionSearchProperties.search().globalSearch().sourceFields(),
                discussionSearchProperties.search().globalSearch().topHitsLimit(),
                discussionSearchProperties.search().globalSearch().aggregation(),
                discussionMapper::toGlobalSearchDataDto);

        if (searchResultAggregatedByType.isEmpty()) {
            return List.of();
        } else {
            List<OutputGlobalSearchDto> globalSearchResults = new ArrayList<>();
            searchResultAggregatedByType.forEach((type, results) ->
                    globalSearchResults.add(OutputGlobalSearchDto
                            .builder()
                            .category(Category.DISCUSSION.getId())
                            .type(type)
                            .data(results)
                            .build()));
            return globalSearchResults;
        }
    }

    @Override
    public void onApplicationEvent(@NonNull UserProfileUpdatedEvent event) {
        try {
            if (event.isUserProfileMinimalUpdated()) {
                syncDocumentsForUserProfile(event.getUserProfile());
            }
        } catch (Exception ex) {
            log.error("Error occurred while syncing events for user profile with id - {}",
                    event.getUserProfile().getId(), ex);
        }
    }

    @NonNull
    @Override
    public Page<Discussion> searchDocumentsForUserProfileSync(@NonNull SearchRequest searchRequest, @NonNull Pageable pageable) {
        return search(searchRequest, null, Function.identity(), pageable);
    }

    @Override
    public void syncDocumentForUserProfile(@NonNull Discussion discussion,
                                           @NonNull UserProfile userProfile) {
        boolean updated = false;

        if (discussion.getUserInfoDTO() != null &&
                Objects.equals(discussion.getUserInfoDTO().getId(), userProfile.getId()) &&
                isDiscussionUserProfileUpdated(discussion.getUserInfoDTO(), userProfile)) {
            updateDiscussionUserInfo(discussion.getUserInfoDTO(), userProfile);
            updated = true;
        }

        if (updated) {
            discussionRepository.save(discussion);
        }
    }

    @NonNull
    @Override
    public Class<Discussion> getDocumentClass() {
        return Discussion.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return discussionSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return discussionRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return discussionRepository;
    }

    @NonNull
    @Override
    public Optional<String> getCreatorUserIdFieldName() {
        return Optional.of(CREATOR_USER_ID_FIELD_NAME);
    }

    @Override
    public @org.jspecify.annotations.NonNull PostProcessFilterResult postProcessQueryFilter(@org.jspecify.annotations.NonNull QueryFilter queryFilter) {
        // skip if user added any filter on visibility
        if (FILTER_KEY_VISIBILITY.equals(queryFilter.fieldName())) {
            return PostProcessFilterResult.builder().processed(true).skip(true).build();
        } else {
            return DiscussionService.super.postProcessQueryFilter(queryFilter);
        }
    }

    private List<QueryFilter> addDefaultFilters(@NonNull List<QueryFilter> queryFilters) {
        List<QueryFilter> filtersWithDefaults = new ArrayList<>(queryFilters);
        // Add default filter to only show the public posts
        filtersWithDefaults.add(QueryFilter
                .builder()
                .fieldName(FILTER_KEY_VISIBILITY)
                .filterValues(FILTER_VISIBILITY_DEFAULT_VALUE)
                .build());
        return filtersWithDefaults;
    }

    private boolean isDiscussionUserProfileUpdated(@NonNull Discussion.UserInfo userInfo,
                                                   @NonNull UserProfile userProfile) {
        return !Objects.equals(userInfo.getFirstName(), userProfile.getFirstName())
                || !Objects.equals(userInfo.getLastName(), userProfile.getLastName())
                || !Objects.equals(userInfo.getEmail(), userProfile.getEmail())
                || !Objects.equals(userInfo.getImageUrl(), userProfile.getImageUrl())
                || !Objects.equals(userInfo.getShortBio(), userProfile.getShortBio())
                || !Objects.equals(userInfo.getOrcid(), userProfile.getOrcid());
    }

    private void updateDiscussionUserInfo(@NonNull Discussion.UserInfo userInfo,
                                          @NonNull UserProfile userProfile) {
        userInfo.setFirstName(userProfile.getFirstName());
        userInfo.setLastName(userProfile.getLastName());
        userInfo.setEmail(userProfile.getEmail());
        userInfo.setImageUrl(userProfile.getImageUrl());
        userInfo.setShortBio(userProfile.getShortBio());
        userInfo.setOrcid(userProfile.getOrcid());
    }

}
