package com.finconsgroup.itserr.marketplace.search.bs.messaging;

import com.finconsgroup.itserr.messaging.consumer.handler.ConsumerMessageHandler;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.utils.CloudEventUtil;
import io.cloudevents.CloudEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

/**
 * Abstract base class for handling CloudEvent messages related to resources.
 * <p>
 * Subclasses must implement {@link #handleResource(String, Object, String)} to define specific processing logic.
 * </p>
 * This class extracts the resource ID and data from the CloudEvent and delegates processing to the subclass.
 */
@Slf4j
public abstract class AbstractResourceSubscriber implements ConsumerMessageHandler {

    protected abstract void handleResource(String id, Object eventData, String eventType);

    /**
     * Handles an incoming CloudEvent message by extracting and processing the associated resource data.
     *
     * @param cloudEvent the CloudEvent containing the resource information
     * @throws WP2MessagingException if message processing fails or if the event is malformed
     */
    @Override
    public void handleMessage(CloudEvent cloudEvent) {
        try {
            if (!StringUtils.hasText(cloudEvent.getId())) {
                throw new IllegalArgumentException("resource Id can not be null");
            }
            log.info("Received message for ingestion with cloudEventId: {}", cloudEvent.getId());
            Object eventData = CloudEventUtil.extractData(cloudEvent);
            EntityIdWrapper entityIdWrapper = CloudEventUtil.getObjectMapper()
                    .convertValue(eventData, EntityIdWrapper.class);
            if (StringUtils.hasText(entityIdWrapper.getId())) {
                handleResource(entityIdWrapper.getId(), eventData, cloudEvent.getType());
            } else {
                throw new IllegalArgumentException("resource entity Id can not be null");
            }
        } catch (Exception e) {
            log.info("handleMessage error: {}", e.getMessage());
            throw new WP2MessagingException(e);
        }
    }

}