package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.extractor;

import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.PlaceholderConfig;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.exception.FieldExtractionException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <p>Registry responsible for managing and coordinating multiple {@link FieldExtractor} instances.
 * This class provides functionality to extract field values from various data sources by delegating the extraction task to the appropriate
 * {@link FieldExtractor} implementation.</p>
 *
 * <p>It attempts to use multiple extractors in order, returning the value extracted by the first
 * extractor that successfully handles the request. If no {@link FieldExtractor} can handle the extraction, an exception is thrown.</p>
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class FieldExtractorRegistry {

    /** A list of {@link FieldExtractor} instances used to extract field values from various data sources. */
    private final List<FieldExtractor> extractors;

    /**
     * <p>Extracts a value from the specified source object based on the provided field name.</p>
     * <p>The method will use all available {@link FieldExtractor}, returning the first successful result, or throwing an {@link UnsupportedOperationException}
     * if no extractor may be found.</p>
     *
     * @param field the name/pattern of the field that should be extracted
     * @param source the object from which the field value should be extracted
     * @return the list of extracted String values, or empty list if the field is not found
     * @throws UnsupportedOperationException when field format or source are not supported.
     */
    @NonNull
    public List<String> extract(
            @NonNull String field,
            @NonNull Object source) {

        Exception lastException = null;

        // Return the first valid result from one of the extractors
        for (final FieldExtractor extractor : extractors) {
            try {
                // If the extraction is successful, return it (first wins)
                return extractor.extract(field, source);
            } catch (final UnsupportedOperationException e) {
                // This extractor does not support the inputs, go to next
            } catch (final Exception e) {
                log.error("Error extracting field '{}'", field, e);
                lastException = e;
            }
        }

        // Throw the last exception (excluding UnsupportedOperationException)
        if (lastException != null) {
            throw new FieldExtractionException(field, source, lastException);
        }

        // No extractor found
        throw new UnsupportedOperationException("No extractor found for field '" + field + "' and given source");

    }

    /**
     * <p>Same as {@link #extract(String, Object)}, but returns a placeholder-compatible value.</p>
     * <p>If a single value is extracted, it is returned directly. If multiple values are extracted, they are joined into a single comma-separated string. If
     * no values are found, null is returned.</p>
     *
     * @param field the name/pattern of the field that should be extracted
     * @param source the object from which the field value should be extracted
     * @return the list of extracted String values, or empty list if the field is not found
     * @throws UnsupportedOperationException when field format or source are not supported.
     */
    public String extractPlaceholderValue(
            @NonNull final String field,
            @NonNull final Object source) {

        final List<String> values = this.extract(field, source);

        if (values.isEmpty()) {
            return null;
        } else if (values.size() == 1) {
            return values.getFirst();
        } else {
            return StringUtils.join(values, ", ");
        }
    }

    /**
     * Extracts placeholder values from a given source object based on a mapping of placeholder names to field names. Uses the provided mapping, retrieves the
     * associated value from the source for each field, and returns a map of placeholder names to their resolved values. Only non-null extracted values are
     * included in the result.
     *
     * @param placeholdersFieldsMap a map where keys are placeholder names and values are field names or patterns to extract from the source
     * @param source the object from which field values will be extracted
     * @return a map containing placeholder names as keys and their extracted string values as values; excludes placeholders with null values
     */
    public Map<String, String> extractPlaceholders(
            @NonNull final Map<String, String> placeholdersFieldsMap,
            @NonNull final Object source) {

        final Map<String, String> placeholders = new HashMap<>();

        for (Map.Entry<String, String> entry : placeholdersFieldsMap.entrySet()) {
            final String placeholderName = entry.getKey();
            final String field = entry.getValue();
            final String value = extractPlaceholderValue(field, source);
            if (value != null) {
                placeholders.put(placeholderName, value);
            }
        }

        return placeholders;

    }

    /**
     * Extracts placeholder values from a given source object based on a mapping of placeholder names to {@link PlaceholderConfig}.
     * Uses the provided configuration to extract values and apply optional mappings for value transformation.
     *
     * @param placeholderConfigs a map where keys are placeholder names and values are {@link PlaceholderConfig} instances
     * @param source the object from which field values will be extracted
     * @return a map containing placeholder names as keys and their extracted (and optionally mapped) string values; excludes placeholders with null values
     */
    public Map<String, String> extractPlaceholdersWithMappings(
            @NonNull final Map<String, PlaceholderConfig> placeholderConfigs,
            @NonNull final Object source) {

        final Map<String, String> placeholders = new HashMap<>();

        for (Map.Entry<String, PlaceholderConfig> entry : placeholderConfigs.entrySet()) {
            final String placeholderName = entry.getKey();
            final PlaceholderConfig config = entry.getValue();
            final String extractedValue = extractPlaceholderValue(config.getPath(), source);
            final String mappedValue = config.applyMapping(extractedValue);
            if (mappedValue != null) {
                placeholders.put(placeholderName, mappedValue);
            }
        }

        return placeholders;

    }

    /**
     * Extracts a set of unique, non-null receivers associated with the specified field from the provided source object.
     *
     * @param field the name or pattern of the field from which values should be extracted
     * @param source the source object containing the field values to be extracted
     * @return a set of unique, non-null extracted receivers associated with the field
     */
    @NonNull
    public Set<String> extractReceivers(
            @NonNull final String field,
            @NonNull final Object source) {

        return extract(field, source)
                .stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

    }

    /**
     * Extracts a single  notification linked resource identifier from the provided source object based on the specified field, if one exists. The first
     * non-null extracted value is returned, or null if no value is found.
     *
     * @param field the name or pattern of the field from which the resource identifier should be extracted; must not be null
     * @param source the object containing the field values to be extracted; must not be null
     * @return the first non-null extracted resource identifier as a string, or null if no value is found
     */
    @Nullable
    public String extractResourceId(
            @NonNull final String field,
            @NonNull final Object source) {

        return extract(field, source)
                .stream()
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null);

    }

}
