package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties;

import jakarta.validation.constraints.NotBlank;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.validation.annotation.Validated;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Base class for notification properties configuration. Serves as a parent class for specific notification property implementations.
 */
@Data
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
@Validated
public class NotificationProperties {

    /** Notification type code. */
    @NotBlank
    private String type;

    /**
     * <p>Configures the source from which the linked resource id should be taken.</p>
     * <p>Allowed values are:
     * <ul>
     * <li><b>$.*</b>: if the source data is a JSON, contains the path the information should be taken from.<br>For example <code>$.user.id</code>, will return the id of the user object</li>
     * </ul>
     * </p>
     */
    @Nullable
    private String resource;

    /**
     * <p>Configures the source from which the receivers should be taken.</p>
     * <p>Allowed values are:
     * <ul>
     * <li><b>$.*</b>: if the source data is a JSON, contains the path the information should be taken from.<br>For example <code>$.user.id</code>, will return the id of the user object</li>
     * </ul>
     * </p>
     */
    @NotBlank
    private String receivers;

    /**
     * <p>A map of placeholder key-value pairs used to dynamically populate notification content.</p>
     * <p>The keys represent placeholder identifiers, and the values can be either:</p>
     * <ul>
     * <li>A simple string with a JSON path (e.g., {@code $.user.username})</li>
     * <li>A complex object with {@code path} and optional {@code mappings} for value transformation</li>
     * </ul>
     * <p>Example YAML configuration:</p>
     * <pre>
     * placeholders:
     *   creator: $.user.name # Simple format
     *   approved: # Complex format with a mapping
     *     path: $.additionalData.approved
     *     mappings:
     *       "true": approved
     *       "false": rejected
     * </pre>
     */
    @NonNull
    @Builder.Default
    private Map<String, Object> placeholders = new HashMap<>();

    /**
     * Returns normalized placeholder configurations from the raw placeholders map.
     * <p>
     * This method handles both simple string values (JSON paths) and complex objects
     * with path and mappings configuration.
     * </p>
     *
     * @return a map of placeholder names to their {@link PlaceholderConfig} instances
     */
    @NonNull
    @SuppressWarnings("unchecked")
    public Map<String, PlaceholderConfig> getNormalizedPlaceholders() {
        final Map<String, PlaceholderConfig> normalized = new LinkedHashMap<>();

        for (Map.Entry<String, Object> entry : placeholders.entrySet()) {
            final String key = entry.getKey();
            final Object value = entry.getValue();

            if (value instanceof String stringValue) {
                // Simple format: just a path string
                normalized.put(key, PlaceholderConfig.builder()
                        .path(stringValue)
                        .build());
            } else if (value instanceof Map<?, ?> mapValue) {
                // Complex format: object with a path, optional mappings, and optional defaultValue
                final Map<String, Object> configMap = (Map<String, Object>) mapValue;
                final String path = (String) configMap.get("path");
                final Map<String, String> mappings = (Map<String, String>) configMap.get("mappings");
                final String defaultValue = (String) configMap.get("defaultValue");

                normalized.put(key, PlaceholderConfig.builder()
                        .path(path)
                        .mappings(mappings)
                        .defaultValue(defaultValue)
                        .build());
            }
        }

        return normalized;
    }

}
