package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config;

import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.BusDriverProperties;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.BusNotificationProperties;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.DriversProperties;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.NotificationsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.JsonNotificationConsumerMessageHandler;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.JsonNotificationConsumerMessageHandlerFactory;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.StreamMultiplexerConsumerMessageHandler;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.StreamMultiplexerConsumerMessageHandlerFactory;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.helper.MessagingHelper;
import com.finconsgroup.itserr.messaging.autoconfigure.MessagingConsumerProperties;
import com.finconsgroup.itserr.messaging.autoconfigure.MessagingStreamProperties;
import com.finconsgroup.itserr.messaging.consumer.MessageConsumer;
import com.finconsgroup.itserr.messaging.enums.OffsetType;
import jakarta.annotation.PostConstruct;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.annotation.Configuration;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Configuration class responsible for setting up and managing notification handlers for processing bus notifications received through messaging streams.
 */
@Slf4j
@Configuration
@RequiredArgsConstructor
public class NotificationConfiguration {

    /** Configuration properties for notifications system */
    private final NotificationsConfigurationProperties notificationsConfig;

    /** Helper for messaging operations */
    private final MessagingHelper messagingHelper;

    /** Factory for creating stream multiplexer message handlers */
    private final StreamMultiplexerConsumerMessageHandlerFactory streamMultiplexerConsumerMessageHandlerFactory;
    /** Factory for creating JSON bus notification message handlers */
    private final JsonNotificationConsumerMessageHandlerFactory jsonNotificationConsumerMessageHandlerFactory;

    /**
     * Record holding the name and properties for a bus notification configuration.
     *
     * @param name The name identifier of the notification
     * @param properties The configuration properties for the notification
     */
    private record NamedBusNotificationProperties(
            String name,
            BusNotificationProperties properties ) {
    }

    /**
     * Initializes the notification configuration by setting up message handlers for configured bus notifications.
     */
    @PostConstruct
    public void initialize() {

        final List<NamedBusNotificationProperties> busNotifications = getBusNotifications();
        if (!busNotifications.isEmpty()) {

            this.configure(busNotifications);

        }

    }

    @NotNull
    private List<NamedBusNotificationProperties> getBusNotifications() {

        return Optional.ofNullable(notificationsConfig)
                .map(NotificationsConfigurationProperties::getDrivers)
                .map(DriversProperties::getBus)
                .map(BusDriverProperties::getNotifications)
                .orElseGet(HashMap::new)
                .entrySet()
                .stream()
                .map(e -> new NamedBusNotificationProperties(e.getKey(), e.getValue()))
                .toList();

    }

    private void configure(
            final Collection<NamedBusNotificationProperties> busNotifications) {

        // Get all unique streams to listen at
        final Set<String> streams = busNotifications.stream()
                .map(NamedBusNotificationProperties::properties)
                .map(BusNotificationProperties::getStream)
                .collect(Collectors.toSet());

        // Initialize streams listening, building and connecting handlers
        final Map<String, StreamMultiplexerConsumerMessageHandler> streamHandlerMap = streams.stream()
                .collect(Collectors.toMap(
                        stream -> stream,
                        this::configure));

        // Initialize notifications handlers
        busNotifications
                .forEach(notification -> configure(
                        streamHandlerMap.get(notification.properties.getStream()),
                        notification));

    }

    private StreamMultiplexerConsumerMessageHandler configure(final String stream) {

        log.debug("Configuring listeners for stream: {}", stream);

        final StreamMultiplexerConsumerMessageHandler notificationsHandler = streamMultiplexerConsumerMessageHandlerFactory.create(stream);

        // TODO: move defaults to properties (to be implemented in SCRUM-1462)
        final MessagingConsumerProperties consumerProperties = MessagingConsumerProperties.builder()
                .enabled(true)
                .name("notification-feeder-" + stream)
                .handlerBeanName("busConsumer")
                .concurrency(1)
                .retryAttempts(3)
                .retryBackoffMs(1000)
                .offsetStrategy(OffsetType.NEXT)
                .singleActiveConsumer(false)
                .dlq(notificationsConfig.getDrivers().getBus().getConsumer().getDlq())
                .build();

        final MessagingStreamProperties streamProperties = MessagingStreamProperties.builder()
                .name(stream)
                .consumer(consumerProperties)
                .build();

        final List<MessageConsumer> consumers = messagingHelper.createConsumers(streamProperties, Map.of("busConsumer", notificationsHandler));
        consumers.forEach(messagingHelper::register);

        return notificationsHandler;

    }

    private void configure(
            final StreamMultiplexerConsumerMessageHandler streamHandler,
            final NamedBusNotificationProperties notification) {

        log.debug("Configuring handler for notification: {}", notification.name);
        final JsonNotificationConsumerMessageHandler notificationHandler = jsonNotificationConsumerMessageHandlerFactory.create(
                notification.name,
                notification.properties);

        streamHandler.addHandler(notificationHandler);

    }

}
