package com.finconsgroup.itserr.marketplace.notification.dm.service.impl;

import com.finconsgroup.itserr.marketplace.notification.dm.configuration.property.TemplateCacheConfigurationProperties;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationEmailTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.NotificationEmailTemplateEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.NotificationTemplateEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.mapper.NotificationEmailTemplateMapper;
import com.finconsgroup.itserr.marketplace.notification.dm.mapper.NotificationTemplateMapper;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.NotificationEmailTemplateRepository;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.NotificationTemplateRepository;
import com.finconsgroup.itserr.marketplace.notification.dm.service.NotificationTemplateService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Default implementation of {@link NotificationTemplateService} to perform operations related to notification templates.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultNotificationTemplateService implements NotificationTemplateService {

    private final TemplateCacheConfigurationProperties cacheConfig;

    private final NotificationTemplateRepository notificationTemplateRepository;

    private final NotificationTemplateMapper notificationTemplateMapper;

    private final NotificationEmailTemplateRepository notificationEmailTemplateRepository;

    private final NotificationEmailTemplateMapper notificationEmailTemplateMapper;

    private final Map<TemplateId, OutputNotificationTemplateDto> templateCache = new HashMap<>();
    private Long templateCacheLastLoad = null;

    private final Map<String, OutputNotificationEmailTemplateDto> emailTemplateCache = new HashMap<>();
    private Long emailTemplateCacheLastLoad = null;

    /**
     * Represents a unique identifier for a specific notification template, based on type and language code.
     */
    private record TemplateId(
            String type,
            String languageCode) {

        public TemplateId(
                final String type,
                final String languageCode) {
            this.type = type;
            this.languageCode = StringUtils.lowerCase(StringUtils.trim(languageCode));
        }
    }

    /**
     * <p>
     * Loads notification templates from the database into an in-memory cache.
     * </p>
     * <p>
     * This method retrieves all {@link NotificationTemplateEntity} objects from the database and maps them to DTO representations using the
     * {@link NotificationTemplateMapper}. The resulting mappings are stored in a synchronized cache that is indexed by {@link TemplateId}, which combines the
     * notification type and language code.
     * </p>
     * <p>
     * After loading the templates, the cache is cleared and repopulated with the newly fetched data. The timestamp of the last load operation is updated to the
     * current system time.
     * </p>
     * <p>
     * Thread safety:<br> The cache modification is wrapped within a synchronized block to ensure thread safety when multiple threads access or modify the cache
     * concurrently.
     * </p>
     */
    private void loadNotificationTemplates() {

        // Load templates from db
        log.debug("Loading notification templates from database");
        final List<NotificationTemplateEntity> templates = notificationTemplateRepository.findAll();

        // Set cache
        synchronized (templateCache) {
            templateCache.clear();
            templates.forEach(
                    t -> templateCache.put(
                            new TemplateId(t.getNotificationType(), t.getLanguageCode()),
                            notificationTemplateMapper.toDto(t)));
        }

        // Set last load
        templateCacheLastLoad = System.currentTimeMillis();
        log.debug("Notification templates loaded successfully");

    }

    @NonNull
    @Override
    public Optional<OutputNotificationTemplateDto> findByTypeAndLanguage(
            @NonNull final String notificationType,
            @NonNull final String languageCode) {

        final TemplateId templateId = new TemplateId(notificationType, languageCode);
        final Integer cacheTimeoutSec = cacheConfig.getTimeout();

        synchronized (templateCache) {

            // Invalidate cache if required
            inValidateCache();

            // Find in cache
            return Optional.ofNullable(templateCache.get(templateId));

        }

    }

    @Override
    public @NonNull List<OutputNotificationTemplateDto> findByType(@NonNull String notificationType) {

        synchronized (templateCache) {

            // Invalidate cache if required
            inValidateCache();

        }

        return templateCache.entrySet().stream()
                .filter(entry -> notificationType.equals(entry.getKey().type()))
                .map(Map.Entry::getValue)
                .toList();
    }

    @Override
    @NonNull
    public List<OutputNotificationEmailTemplateDto> findAllEmailTemplates() {
        synchronized (emailTemplateCache) {

            // Invalidate cache if required
            inValidateEmailCache();

        }

        return emailTemplateCache.values().stream().toList();
    }

    private void inValidateCache() {
        final Integer cacheTimeoutSec = cacheConfig.getTimeout();
        if (templateCacheLastLoad == null ||
                cacheTimeoutSec == null ||
                System.currentTimeMillis() - templateCacheLastLoad > cacheTimeoutSec * 1000) {
            log.debug("Notification templates cache is expired");
            loadNotificationTemplates();
        }
    }

    private void inValidateEmailCache() {
        final Integer cacheTimeoutSec = cacheConfig.getTimeout();
        if (emailTemplateCacheLastLoad == null ||
                cacheTimeoutSec == null ||
                System.currentTimeMillis() - emailTemplateCacheLastLoad > cacheTimeoutSec * 1000) {
            log.debug("Notification email templates cache is expired");
            loadNotificationEmailTemplates();
        }
    }

    /**
     * <p>
     * Loads notification email templates from the database into an in-memory cache.
     * </p>
     * <p>
     * After loading the templates, the cache is cleared and repopulated with the newly fetched data. The timestamp of the last load operation is updated to the
     * current system time.
     * </p>
     * <p>
     * Thread safety:<br> The cache modification is wrapped within a synchronized block to ensure thread safety when multiple threads access or modify the cache
     * concurrently.
     * </p>
     */
    private void loadNotificationEmailTemplates() {

        // Load templates from db
        log.debug("Loading notification email templates from database");
        final List<NotificationEmailTemplateEntity> emailTemplates = notificationEmailTemplateRepository.findAll();

        // Set cache
        synchronized (emailTemplateCache) {
            emailTemplateCache.clear();
            emailTemplates.forEach(
                    t -> emailTemplateCache.put(
                            t.getLanguageCode(),
                            notificationEmailTemplateMapper.toDto(t)));
        }

        // Set last load
        emailTemplateCacheLastLoad = System.currentTimeMillis();
        log.debug("Notification email templates loaded successfully");

    }

}
