package com.finconsgroup.itserr.marketplace.notification.dm.repository.specification;

import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity_;
import org.springframework.data.jpa.domain.Specification;

import java.util.Locale;
import java.util.UUID;

/**
 * Utility class providing reusable JPA specifications for filtering user notifications based on specific criteria.
 */
public final class UserNotificationSpecifications {

    private UserNotificationSpecifications() {
        throw new UnsupportedOperationException("Cannot instantiate");
    }

    /**
     * Creates a specification to filter user notifications based on the user ID.
     *
     * @param userId Id of the user the notifications belong to.
     * @param userUsername Username of the user the notifications belong to.
     * @param userEmail Email of the user the notifications belong to.
     * @return a specification that filters notifications by the given user ID, or a conjunction specification if the user is null.
     */
    public static Specification<UserNotificationEntity> user(
            final UUID userId,
            final String userUsername,
            final String userEmail) {
        return userId(userId)
                .or(userUsername(userUsername))
                .or(userEmail(userEmail));
    }

    /**
     * Creates a specification to filter user notifications based on user ID.
     *
     * @param userId The UUID of the user to filter by
     * @return a specification that filters notifications by user ID, or conjunction specification if userId is null
     */
    public static Specification<UserNotificationEntity> userId(
            final UUID userId) {
        return (root, query, cb) ->
                userId != null
                        ? cb.equal(cb.lower(root.get(UserNotificationEntity_.user)), userId.toString().toLowerCase(Locale.ROOT))
                        : cb.conjunction();
    }

    /**
     * Creates a specification to filter user notifications based on username.
     *
     * @param userUsername The username to filter by
     * @return a specification that filters notifications by username, or conjunction specification if username is null
     */
    public static Specification<UserNotificationEntity> userUsername(
            String userUsername) {
        return (root, query, cb) ->
                userUsername != null
                        ? cb.equal(cb.lower(root.get(UserNotificationEntity_.user)), userUsername.toLowerCase(Locale.ROOT))
                        : cb.conjunction();
    }

    /**
     * Creates a specification to filter user notifications based on user email.
     *
     * @param userEmail The email address to filter by
     * @return a specification that filters notifications by email, or conjunction specification if email is null
     */
    public static Specification<UserNotificationEntity> userEmail(
            String userEmail) {
        return (root, query, cb) ->
                userEmail != null
                        ? cb.equal(cb.lower(root.get(UserNotificationEntity_.user)), userEmail.toLowerCase(Locale.ROOT))
                        : cb.conjunction();
    }

    /**
     * Creates a specification to filter user notifications based on their read status.
     *
     * @param read Boolean value indicating the read status to filter by (true for read, false for unread)
     * @return a specification that filters notifications by read status, or conjunction specification if read is null
     */
    public static Specification<UserNotificationEntity> read(final Boolean read) {
        return (root, query, cb) ->
                read != null
                        ? cb.equal(root.get(UserNotificationEntity_.read), read)
                        : cb.conjunction();
    }

    /**
     * Creates a specification to filter user notifications based on their archived status.
     *
     * @param archived Boolean value indicating the archived status to filter by (true for archived, false for not archived)
     * @return a specification that filters notifications by archived status, or conjunction specification if archived is null
     */
    public static Specification<UserNotificationEntity> archived(final Boolean archived) {
        return (root, query, cb) ->
                archived != null
                        ? cb.equal(root.get(UserNotificationEntity_.archived), archived)
                        : cb.conjunction();
    }

}
