package com.finconsgroup.itserr.marketplace.notification.dm.service.impl;

import com.finconsgroup.itserr.marketplace.notification.dm.service.LanguageService;
import io.micrometer.common.util.StringUtils;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Default implementation of LanguageService that handles language preferences based on HTTP Accept-Language header. Provides functionality to parse and
 * prioritize language preferences while ensuring a default language fallback.
 */
@Component
@RequiredArgsConstructor
public class DefaultLanguageService implements LanguageService {

    /** Default language code used when no language preferences are specified */
    private static final String DEFAULT_LANGUAGE = "en";

    /**
     * Record representing a language preference with its priority from Accept-Language header.
     */
    private record Language(
            String code,
            double priority ) {
    }

    @Override
    public List<String> getPreferredLanguages(final String httpAcceptLanguage) {

        // Initialize
        final Set<String> preferredLanguages = new LinkedHashSet<>();

        // Parse Accept-Language header
        if (httpAcceptLanguage != null) {
            preferredLanguages.addAll(parseAcceptLanguageHeader(httpAcceptLanguage));
        }

        // Add default language if missing
        preferredLanguages.add(DEFAULT_LANGUAGE);

        // Return list
        return preferredLanguages.stream().toList();

    }

    /**
     * Parses the Accept-Language header value into a sorted list of language codes.
     *
     * @param langHeader the Accept-Language header value to parse
     * @return List of language codes sorted by their priority
     */
    private List<String> parseAcceptLanguageHeader(final String langHeader) {
        return Arrays.stream(langHeader.split(","))
                .map(this::parseAcceptLanguageHeaderComponent)
                .filter(Objects::nonNull)
                .sorted((l1, l2) -> -Double.compare(l1.priority, l2.priority))
                .map(l -> l.code)
                .toList();
    }

    /**
     * Parses a single component of the Accept-Language header.
     *
     * @param s the header component to parse
     * @return Language record containing the language code and its priority, or null if invalid
     */
    private Language parseAcceptLanguageHeaderComponent(final String s) {

        // Blank check
        if (StringUtils.isBlank(s)) {
            return null;
        }

        // Split parts
        final String[] parts = s.split(";");

        // Get language code
        final String languageCode = parts[0];

        // Get priority
        double priority = 1.0;
        for (int i = 1; i < parts.length; i++) {
            if (parts[i].startsWith("q=")) {
                try {
                    priority = Double.parseDouble(parts[i].split("=")[1]);
                } catch (Exception e) {
                    // Ignore any invalid request
                }
            }
        }

        // Return final language
        return new Language(languageCode, priority);

    }

}
