/**
 * Global exception handler for REST controllers in the notification system.
 * This class handles custom exceptions related to notification and category operations
 * and returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a NotificationNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.notification.dm.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.NOT_FOUND;

/**
 * Custom Exception Handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles NotificationNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex the NotificationNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({UserNotificationNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleNotificationNotFoundException(
            final UserNotificationNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

}
