package com.finconsgroup.itserr.marketplace.notification.dm.service.impl;

import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.UserDto;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity_;
import com.finconsgroup.itserr.marketplace.notification.dm.exception.UserNotificationNotFoundException;
import com.finconsgroup.itserr.marketplace.notification.dm.mapper.LocalizedUserNotificationMapper;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.UserNotificationRepository;
import com.finconsgroup.itserr.marketplace.notification.dm.repository.specification.UserNotificationSpecifications;
import com.finconsgroup.itserr.marketplace.notification.dm.service.LanguageService;
import com.finconsgroup.itserr.marketplace.notification.dm.service.LocalizedUserNotificationService;
import com.finconsgroup.itserr.marketplace.notification.dm.service.NotificationTemplateService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * Default implementation of {@link LocalizedUserNotificationService} to perform operations related to localized notification resources. Uses {@link UserDto} to
 * represent user information across methods.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultLocalizedUserNotificationService implements LocalizedUserNotificationService {

    /** Repository for managing user notification entities */
    private final UserNotificationRepository userNotificationRepository;

    /** Service for managing notification templates */
    private final NotificationTemplateService notificationTemplateService;

    /** Mapper for converting notification entities to DTOs */
    private final LocalizedUserNotificationMapper localizedUserNotificationMapper;

    /** Utility for managing language preferences */
    private final LanguageService languageService;

    /**
     * Converts a UserNotificationEntity to a localized DTO using the preferred language template.
     *
     * @param entity the notification entity to convert
     * @param httpAcceptLanguage HTTP Accept-Language header. May be null.
     * @return the localized notification DTO
     */
    private OutputLocalizedUserNotificationDto toLocalizedDto(
            final UserNotificationEntity entity,
            final String httpAcceptLanguage) {

        // Get preferred language
        final List<String> preferredLanguages = languageService.getPreferredLanguages(httpAcceptLanguage);

        // Find the best language for which a template exists
        OutputNotificationTemplateDto template = null;
        final String type = entity.getType();
        for (String languageCode : preferredLanguages) {
            Optional<OutputNotificationTemplateDto> templateOpt = notificationTemplateService.findByTypeAndLanguage(type, languageCode);
            if (templateOpt.isPresent()) {
                template = templateOpt.get();
                break;
            }
        }

        // Map the entity and the message (use the message type itself if no template found)
        return localizedUserNotificationMapper.toDto(
                entity,
                template != null ? template.getText() : type);

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputLocalizedUserNotificationDto> findAll(
            @NonNull final UserDto user,
            final Boolean read,
            final Boolean archived,
            @NonNull final Pageable pageable,
            final String httpAcceptLanguage) {

        final Pageable sortedPageable = PageRequest.of(
                pageable.getPageNumber(),
                pageable.getPageSize(),
                Sort.by(Sort.Direction.DESC, UserNotificationEntity_.CREATION_TIME)
        );

        return userNotificationRepository.findAll(
                        UserNotificationSpecifications.user(user.id(), user.username(), user.email())
                                .and(UserNotificationSpecifications.read(read))
                                .and(UserNotificationSpecifications.archived(archived)),
                        sortedPageable)
                .map(n -> this.toLocalizedDto(n, httpAcceptLanguage));

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputLocalizedUserNotificationDto findById(
            @NonNull final UUID notificationId,
            final String httpAcceptLanguage) {

        return userNotificationRepository.findById(notificationId)
                .map(n -> this.toLocalizedDto(n, httpAcceptLanguage))
                .orElseThrow(() -> new UserNotificationNotFoundException(notificationId));

    }

    @NonNull
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputLocalizedUserNotificationDto findByUser(
            @NonNull final UserDto user,
            @NonNull final UUID notificationId,
            final String httpAcceptLanguage) {

        return userNotificationRepository.findByUserAndId(user.id().toString(), user.username(), user.email(), notificationId)
                .map(n -> this.toLocalizedDto(n, httpAcceptLanguage))
                .orElseThrow(() -> new UserNotificationNotFoundException(notificationId));

    }

}
