package com.finconsgroup.itserr.marketplace.notification.dm.mapper;

import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.entity.UserNotificationEntity;
import org.mapstruct.Context;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * Mapper interface for converting {@link UserNotificationEntity} to localized {@link OutputLocalizedUserNotificationDto}. Handles message placeholder
 * resolution during the mapping process.
 */
@Mapper(config = MapperConfiguration.class)
public interface LocalizedUserNotificationMapper {

    /** Pattern to match placeholder syntax in the format {{placeholder}} */
    Pattern PLACEHOLDER_PATTERN = Pattern.compile("\\{\\{(.+?)\\}\\}");

    /**
     * Maps {@link UserNotificationEntity} to {@link OutputLocalizedUserNotificationDto} with resolved message placeholders.
     *
     * @param userNotificationEntity the source notification entity
     * @param message the localized message template
     * @return mapped DTO with resolved message
     */
    @Mapping(target = "message", expression = "java(resolveMessage(message, userNotificationEntity.getPlaceholderValues()))")
    OutputLocalizedUserNotificationDto toDto(
            UserNotificationEntity userNotificationEntity,
            @Context String message);

    /**
     * Resolves message placeholders using provided values map.
     *
     * @param message the message template containing placeholders
     * @param placeholders map of placeholder values
     * @return resolved message with replaced placeholders
     */
    default String resolveMessage(
            final String message,
            @Context final Map<String, String> placeholders) {

        final Map<String, String> actualPlaceholders = placeholders != null
                ? new HashMap<>(placeholders)
                : new HashMap<>();

        return PLACEHOLDER_PATTERN.matcher(message)
                .replaceAll(mr -> actualPlaceholders.getOrDefault(mr.group(1), mr.group(0)));

    }

}
