package com.finconsgroup.itserr.marketplace.notification.dm.service;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.UserDto;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.UUID;

/**
 * Service for handling business logic related to UserNotification. This interface provides methods to retrieve, create, update, and delete UserNotification,
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputUserNotificationDto> UserNotifications = userNotificationService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface UserNotificationService {

    /**
     * Creates a new notification for user using the provided input data.
     *
     * @param inputCreateUserNotificationDto the input DTO containing user notification details
     * @return an {@link OutputUserNotificationDto} representing the created user notification
     */
    @NonNull
    List<OutputUserNotificationDto> create(
            @NonNull InputCreateUserNotificationDto inputCreateUserNotificationDto);

    /**
     * Updates an existing user notification by id with the provided data.
     *
     * @param user the user DTO containing identification information
     * @param notificationId the id of the user notification to update
     * @param inputPatchUserNotificationDto the input DTO containing updated user notification details
     * @return an OutputUserNotificationDto notificationDto representing the updated user notification
     * @throws WP2ResourceNotFoundException if no user notification is found with the given id
     */
    @NonNull
    OutputPatchUserNotificationDto patchById(
            @NonNull UserDto user,
            @NonNull UUID notificationId,
            @NonNull InputPatchUserNotificationDto inputPatchUserNotificationDto);

    /**
     * Deletes a user notification by id. It will move the notification entity on a separated table containing deleted records.
     *
     * @param user the user DTO containing identification information
     * @param notificationId the id of the user notification to delete
     * @throws WP2ResourceNotFoundException if no user notification is found with the given id
     */
    @NonNull
    void deleteById(
            @NonNull UserDto user,
            @NonNull UUID notificationId);

    /**
     * Updates the state of all user notifications based on the provided input.
     *
     * @param user the user DTO containing identification information
     * @param request the input DTO containing the new state to apply to all user notifications.
     */
    void patchAll(
            @NonNull UserDto user,
            @NonNull InputPatchUserNotificationsDto request);

}
