package com.finconsgroup.itserr.marketplace.notification.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationEmailTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputUserNotificationDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Notification Management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, updating, and deleting Notifications.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET  notification/notifications - Retrieve a paginated list of all Notifications.
 * GET  notification/notifications/{id} - Retrieve a notification by id.
 * POST notification/notifications - Create a new notification.
 * PATCH  notification/notifications/{id} - Update an existing notification.
 * DELETE notification/notifications/{id} - Delete a notification.
 * </pre>
 * </p>
 */
@Tag(
        name = "UserNotificationApi",
        description = "The UserNotificationApi API: it provides endpoints for " +
                "retrieving, creating, updating, and deleting Notifications for a user."
)
@SecurityRequirement(name = "BearerAuth")
public interface UserNotificationApi {

    /**
     * Retrieves a paginated list of all user Notifications.
     *
     * @param read           filter notifications by read status (true for read, false for unread, null for all)
     * @param archived       filter notifications by archived status (true for archived, false for not archived, null for all)
     * @param pageNumber     the page number to retrieve (default is 0)
     * @param pageSize       the number of Notifications per page (default is 10)
     * @param acceptLanguage the language code for localization (e.g. "en-US")
     * @return a page of {@link OutputUserNotificationDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all user Notifications",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/notification/notifications", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputLocalizedUserNotificationDto> findAll(
            @RequestParam(name = "read", required = false) Boolean read,
            @RequestParam(name = "archived", required = false) Boolean archived,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestHeader(name = "Accept-Language", required = false) String acceptLanguage
    );

    /**
     * Patches all existing user notifications.
     *
     * @param request the patching request
     */
    @Operation(
            summary = "patch all user notifications, mark notification as read/unread",
            responses = {
                    @ApiResponse(responseCode = "204", description = "OK"),
            }
    )
    @PatchMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void patchAll(
            @Valid @RequestBody InputPatchUserNotificationsDto request
    );

    /**
     * Retrieves a user notification by id.
     *
     * @param id             the id of the notification to retrieve
     * @param acceptLanguage the language code for localization (e.g. "en-US")
     * @return the found {@link OutputUserNotificationDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find user notification by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/notification/notifications/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputLocalizedUserNotificationDto findById(
            @PathVariable("id") UUID id,
            @RequestHeader(name = "Accept-Language", required = false) String acceptLanguage);

    /**
     * Creates a new notification for each argument user.
     *
     * @param inputCreateUserNotificationDto the input data transfer object containing user notification details to create
     * @return the created list of {@link OutputUserNotificationDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create a user notification",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/notification/notifications",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    List<OutputUserNotificationDto> createNotification(
            @Valid @RequestBody InputCreateUserNotificationDto inputCreateUserNotificationDto);

    /**
     * Patches an existing user notification by id.
     *
     * @param id                            the id of the notification to patch
     * @param inputPatchUserNotificationDto the input data transfer object containing patched notification details
     * @return the patched {@link OutputPatchUserNotificationDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "patch notification by id, mark notification as read/unRead or archive/un-archive",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @PatchMapping(
            value = "/notification/notifications/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputPatchUserNotificationDto patchUserNotificationById(
            @PathVariable("id") UUID id,
            @Valid @RequestBody InputPatchUserNotificationDto inputPatchUserNotificationDto
    );

    /**
     * Deletes a user notification by id.
     *
     * @param id the id of the notification to delete
     */
    @Operation(
            summary = "delete a user notification by id",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/notification/notifications/{id}")
    void deleteUserNotificationById(@PathVariable("id") UUID id);


    /**
     * Retrieves user notification message templates.
     *
     * @param notificationType the notification type which will be used as key for the templates
     * @return the found {@link List<OutputNotificationTemplateDto> } and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find notification templates by type",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/notification/notifications/templates/{notificationType}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputNotificationTemplateDto> getTemplatesByType(
            @PathVariable("notificationType") String notificationType);

    /**
     * Retrieves all notification email templates.
     *
     * @return the found {@link List<OutputNotificationEmailTemplateDto> } and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find notification templates by type",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK")
            }
    )
    @GetMapping(value = "/notification/notifications/templates/email", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputNotificationEmailTemplateDto> getEmailTemplates();

}
