package com.finconsgroup.itserr.marketplace.notification.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.notification.bs.client.NotificationDmClient;
import com.finconsgroup.itserr.marketplace.notification.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.InputPatchUserNotificationsDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputNotificationEmailTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputNotificationTemplateDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputPatchUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.notification.bs.event.NotificationPublisher;
import com.finconsgroup.itserr.marketplace.notification.bs.mapper.NotificationMapper;
import com.finconsgroup.itserr.marketplace.notification.bs.service.UserNotificationService;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link UserNotificationService} to perform operations related to notification resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultUserNotificationService implements UserNotificationService {

    private final NotificationDmClient notificationDmClient;

    private final NotificationMapper notificationMapper;

    private final NotificationPublisher notificationPublisher;

    private final UserProfileDmClient userProfileDmClient;

    private final EmailService emailService;


    @Override
    public @NotNull List<OutputUserNotificationDto> create(
            @NotNull InputCreateUserNotificationDto request,
            final String acceptLanguage) {

        // Create notification on bs service
        final List<OutputUserNotificationDto> notifications;

        try {
            log.debug("Creating notification on bs service");
            notifications = notificationDmClient.create(request, acceptLanguage);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }

        // Publish created event for websocket sending
        try {
            log.debug("Publishing notification on websockets");
            for (final OutputUserNotificationDto notification : notifications) {
                notificationPublisher.publishCreated(notificationMapper.toEvent(notification));
            }
        } catch (final Exception e) {
            log.error("Error publishing created notification for websocket handling", e);
        }

        // Below the code for sending emails
        log.info("send email procedure\n ================START================");
        List<OutputNotificationTemplateDto> messageTemplates = notificationDmClient.getMessageTemplatesByType(request.getType());
        if (!CollectionUtils.isEmpty(messageTemplates)) {

            // Creating a map where language code (key) is mapped to template text (value)
            // Example: { "EN" -> "email deleted", "IT" -> "email cancellata" }
            @NotNull Map<String, String> languageTemplateMap = messageTemplates.stream()
                    .filter(template -> template.getLanguageCode() != null && template.getText() != null)
                    .collect(Collectors.toMap(template -> template.getLanguageCode().toUpperCase(), OutputNotificationTemplateDto::getText));

            InputFindUserProfilesByTokenInfoDto inputUserTokenInfoList;
            try {
                log.info("Collecting users details by mixed token info");
                inputUserTokenInfoList = new InputFindUserProfilesByTokenInfoDto(
                        notifications.stream().map(OutputUserNotificationDto::getUser).toList());

                List<OutputUserProfileDto> userProfiles = userProfileDmClient.findAllByTokenInfo(inputUserTokenInfoList);

                //Filtering users for those who gave the consent to receive emails
                @NotNull Set<OutputUserProfileDto> filteredUsers = userProfiles.stream().filter(OutputUserProfileDto::isEmailConsent).collect(Collectors.toSet());

                preparingAndSendingEmails(filteredUsers, languageTemplateMap, notifications);

                log.info("{} emails successfully sent", filteredUsers.size());

            } catch (final Exception e) {
                log.error("Error while publishing emails", e);
            }
            // Below the code for sending emails
            log.info("send email procedure\n ================END================");

        }

        return notifications;
    }

    private void preparingAndSendingEmails(@NotNull Set<OutputUserProfileDto> usersWithEmailConsent,
                                           @NotNull Map<String, String> languageTemplateMap,
                                           List<OutputUserNotificationDto> notifications) {

        List<OutputNotificationEmailTemplateDto> emailTemplates = notificationDmClient.getEmailTemplates();
        Map<String, OutputNotificationEmailTemplateDto> languageEmailTemplateMap = emailTemplates.stream()
                .filter(dto -> dto.getLanguageCode() != null)
                .collect(Collectors.toMap(dto -> dto.getLanguageCode().toUpperCase(), Function.identity()));

        // Iterate through each notification
        for (OutputUserNotificationDto notification : notifications) {
            try {

                // Find matching user profile by id, email, or username
                Optional<OutputUserProfileDto> matchedUser = usersWithEmailConsent.stream()
                        .filter(user -> matchesUserIdentifier(user, notification.getUser()))
                        .findFirst();

                if (matchedUser.isEmpty()) {
                    log.warn("No user profile found for notification user identifier: {}",
                            notification.getUser());
                    continue;
                }

                OutputUserProfileDto userProfile = matchedUser.get();

                // Get message template based on user's preferred language if available
                String template = languageTemplateMap.get(userProfile.getLanguage().toUpperCase());
                if (template == null || template.isEmpty()) {
                    log.warn("No message template found for language: {}, user: {} , adding english language as default",
                            userProfile.getLanguage(), userProfile.getEmail());

                    template = languageTemplateMap.get("EN");
                }

                // Get email template based on user's preferred language if available
                OutputNotificationEmailTemplateDto emailTemplate = languageEmailTemplateMap.get(userProfile.getLanguage().toUpperCase());
                if (emailTemplate == null) {
                    log.warn("No email template found for language: {}, user: {} , adding english language as default",
                            userProfile.getLanguage(), userProfile.getEmail());

                    emailTemplate = languageEmailTemplateMap.get("EN");
                }

                // Replace placeholders in template with actual values
                String notificationMessage = replacePlaceholders(template, notification.getPlaceholderValues());

                // Generate email subject from notification type
                String subject = extractSubjectFromNotificationType(notification.getType(), emailTemplate.getSubjectText());

                // Construct username
                String userName = String.format("%s %s", userProfile.getFirstName(), userProfile.getLastName());

                // Construct email body
                String emailBody = constructEmailBody(userName, notificationMessage, emailTemplate.getBodyText());

                // Send email
                boolean sent = emailService.sendEmail(userProfile.getEmail(), subject, emailBody);

                if (sent) {
                    log.info("Email notification sent successfully to: {} for type: {}",
                            userProfile.getEmail(), notification.getType());
                } else {
                    log.error("Failed to send email notification to: {} for type: {}",
                            userProfile.getEmail(), notification.getType());
                }

            } catch (Exception e) {
                log.error("Error processing notification for user: {}", notification.getUser(), e);
            }
        }
    }


    /**
     * Checks if a user profile matches the given identifier
     *
     * @param user       the user profile to check
     * @param identifier the identifier to match against (UUID, email, or username)
     * @return true if the user matches the identifier
     */
    private boolean matchesUserIdentifier(OutputUserProfileDto user, String identifier) {
        // Try matching as UUID (id)
        try {
            UUID uuid = UUID.fromString(identifier);
            if (user.getId() != null && user.getId().equals(uuid)) {
                return true;
            }
        } catch (IllegalArgumentException e) {
            // Not a valid UUID - just ignore
        }

        // Try matching as email
        if (user.getEmail() != null && user.getEmail().equalsIgnoreCase(identifier)) {
            return true;
        }

        // Try matching as username
        if (user.getPreferredUsername() != null &&
                user.getPreferredUsername().equalsIgnoreCase(identifier)) {
            return true;
        }

        return false;
    }

    /**
     * Replaces placeholders in the template with actual values.
     * Placeholders are expected in the format: {{placeholderKey}}
     * Matching is case-insensitive and removes surrounding quotes/apostrophes.
     * <p>
     * Examples:
     * - '{{title}}' or "{{Title}}" or {{TITLE}} → replaced with value
     *
     * @param template     the email template containing placeholders
     * @param placeholders map of placeholder keys and their replacement values
     * @return the template with all placeholders replaced
     */
    private String replacePlaceholders(String template, Map<String, String> placeholders) {
        if (placeholders == null || placeholders.isEmpty()) {
            return template;
        }

        String result = template;

        for (Map.Entry<String, String> entry : placeholders.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue() != null ? entry.getValue() : "";

            // Pattern matches: ['"]?{{key}}['"]? (case-insensitive)
            // This handles: {{key}}, '{{key}}', "{{key}}", {{Key}}, etc.
            String patternString = "['\"]?\\{\\{" + Pattern.quote(key) + "\\}\\}['\"]?";
            Pattern pattern = Pattern.compile(patternString, Pattern.CASE_INSENSITIVE);

            Matcher matcher = pattern.matcher(result);
            result = matcher.replaceAll(Matcher.quoteReplacement(value));
        }

        return result;
    }

    /**
     * Extracts a human-readable email subject from notification type
     * Converts "catalog-item-event-created" to "Catalog Item Event Created"
     *
     * @param notificationType the notification type identifier
     * @return formatted email subject
     */
    private String extractSubjectFromNotificationType(String notificationType, String defaultNotificationType) {
        if (notificationType == null || notificationType.isEmpty()) {
            return defaultNotificationType == null || defaultNotificationType.isEmpty()
                    ? "Notification"
                    : defaultNotificationType;
        }

        // Split by dash and capitalize each word
        return Arrays.stream(notificationType.split("-"))
                .map(word -> word.substring(0, 1).toUpperCase() + word.substring(1))
                .collect(Collectors.joining(" "));
    }

    public String constructEmailBody(String userName, String notificationMessage, String bodyText) {
        try {
            return bodyText
                    .replace("{{name}}", userName)
                    .replace("{{message}}", notificationMessage);
        } catch (Exception e) {
            log.error("Failed to construct email body", e);
        }
        return notificationMessage;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public @NotNull OutputPatchUserNotificationDto patchById(
            @NotNull UUID notificationId,
            @NotNull InputPatchUserNotificationDto inputPatchUserNotificationDto) {
        try {
            return notificationDmClient.patchById(notificationId, inputPatchUserNotificationDto);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(notificationId);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(@NotNull UUID notificationId) {
        try {
            notificationDmClient.deleteById(notificationId);
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(notificationId);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

    @Override
    public void patchAll(@NonNull final InputPatchUserNotificationsDto request) {
        try {
            notificationDmClient.patchAll(request);
        } catch (Exception e) {
            throw new WP2ExecutionException(e);
        }
    }

}
