package com.finconsgroup.itserr.marketplace.news.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.news.dm.entity.converter.StringListConverter;
import com.finconsgroup.itserr.marketplace.news.dm.model.InstitutionalPageModel;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.Lob;
import jakarta.persistence.OneToMany;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

/**
 * Entity class representing news in the news system.
 * This entity is mapped to the "news" table in the database and stores
 * information such as the ID, title, news type, content and authors.
 *
 * <p>Example usage:
 * <pre>
 * NewsEntity news = NewsEntity.builder()
 *     .title("news title")
 *     .newsType("catalog-item-created")
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "news")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class NewsEntity extends AbstractUUIDEntity {

    /**
     * Title field length.
     */
    public static final int TITLE_LENGTH = 200;
    /**
     * Type field length.
     */
    public static final int TYPE_LENGTH = 50;
    /**
     * Content field length.
     */
    public static final int CONTENT_LENGTH = 8000;


    /**
     * The title of the news.
     */
    @Column(name = "title", nullable = false, length = TITLE_LENGTH)
    private String title;

    /**
     * The ID of the user who created the news.
     */
    @Column(name = "creator_id", nullable = false)
    private UUID creatorId;

    /**
     * The type of news.
     */
    @Column(name = "news_type", length = TYPE_LENGTH)
    private String newsType;

    /**
     * The content of the news.
     */
    @Column(name = "content", length = CONTENT_LENGTH)
    private String content;

    /**
     * List of tags associated with the news.
     */
    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "tags")
    private List<String> tags;

    /**
     * The JSON structure for Institutional Page.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "institutional_pages", columnDefinition = "json")
    private List<InstitutionalPageModel> institutionalPages;

    /**
     * The document ID of the image.
     */
    @Column(name = "image_id")
    private UUID imageId;

    /**
     * Public URL of the image.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * List of registered author IDs.
     */
    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "registered_author_ids")
    private List<String> registeredAuthorIds;

    /**
     * List of unregistered author.
     */
    @OneToMany(mappedBy = "news", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<AuthorEntity> unRegisteredAuthors;

    /**
     * The timestamp when the news was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the news was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }

}
