package com.finconsgroup.itserr.marketplace.news.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.dm.dto.OutputNewsDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to News Management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, updating, and deleting news.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET  news/news - Retrieve a paginated list of all news.
 * GET  news/news/{id} - Retrieve a news by id.
 * POST news/news - Create a new news.
 * PUT  news/news/{id} - Update an existing news.
 * DELETE news/news/{id} - Delete a news.
 * </pre>
 * </p>
 */
@Tag(
        name = "NewsApi",
        description = "The NewsApi API: it provides endpoints for " +
                "retrieving, creating, updating, and deleting news."
)
@SecurityRequirement(name = "BearerAuth")
public interface NewsApi {

    /**
     * Retrieves a paginated list of all news.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "startTime")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of {@link OutputNewsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all the news",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/news/news", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputNewsDto> findAll(
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "creationTime", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves news by id.
     *
     * @param id the id of the news to retrieve
     * @return the found {@link OutputNewsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find news by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/news/news/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputNewsDto findById(@PathVariable("id") UUID id);

    /**
     * Creates new news.
     *
     * @param inputCreateNewsDto the input data transfer object containing news details
     */
    @Operation(
            summary = "create a news",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/news/news",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputNewsDto createNews(@Valid @RequestBody InputCreateNewsDto inputCreateNewsDto);

    /**
     * Updates existing news by id.
     *
     * @param id                 the id of the news to update
     * @param inputUpdateNewsDto the input data transfer object containing updated news details
     * @return the updated {@link OutputNewsDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "updates news by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @PutMapping(
            value = "/news/news/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputNewsDto updateNewsById(
            @PathVariable("id") UUID id,
            @Valid @RequestBody InputUpdateNewsDto inputUpdateNewsDto
    );

    /**
     * Deletes news by id.
     *
     * @param id the id of the news to delete
     */
    @Operation(
            summary = "delete a news by id",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/news/news/{id}")
    void deleteNewsById(@PathVariable("id") UUID id);

}
