package com.finconsgroup.itserr.marketplace.news.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.news.bs.api.DiagnosticsApi;
import com.finconsgroup.itserr.marketplace.news.bs.bean.NewsApplicationEvent;
import com.finconsgroup.itserr.marketplace.news.bs.config.properties.NewsBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsDetailDto;
import com.finconsgroup.itserr.marketplace.news.bs.enums.EventType;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    private final ApplicationEventPublisher applicationEventPublisher;
    private final NewsBsConfigurationProperties newsBsConfigurationProperties;

    // For storing rabbitmq test messages
    private static final Map<String, OutputNewsDetailDto> itemMap = new HashMap<>();

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    // private

    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

    @Override
    public ResponseEntity<String> publishMessage(EventType eventType, String itemName) {
        if (!newsBsConfigurationProperties.getDiagnostics().isPublishMessageEnabled()) {
            log.warn("Attempt to access disabled diagnostics endpoint");
            return ResponseEntity.status(HttpStatus.FORBIDDEN)
                    .body("This endpoint is disabled in the current environment");
        }

        log.debug("call to DiagnosticsController - publishMessage, name: %s".formatted(itemName));

        OutputNewsDetailDto newsDetailDto;
        switch (eventType) {
            case CREATED:
                newsDetailDto = OutputNewsDetailDto.builder()
                        .id(UUID.randomUUID())
                        .title(itemName)
                        .content("CREATED News content")
                        .build();
                log.info("sending created event message, newsDetailDto: %s".formatted(newsDetailDto));
                itemMap.put(newsDetailDto.getTitle(), newsDetailDto);
                applicationEventPublisher.publishEvent(new NewsApplicationEvent(newsDetailDto, EventType.CREATED));
                break;

            case UPDATED:
                newsDetailDto = itemMap.get(itemName);
                if (newsDetailDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                newsDetailDto.setContent("the UPDATED content");
                log.info("sending updated event message, newsDetailDto: %s".formatted(newsDetailDto));
                applicationEventPublisher.publishEvent(new NewsApplicationEvent(newsDetailDto, EventType.UPDATED));
                break;

            case DELETED:
                newsDetailDto = itemMap.get(itemName);
                if (newsDetailDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                log.info("sending deleted message, newsDetailDto: %s".formatted(newsDetailDto));
                applicationEventPublisher.publishEvent(new NewsApplicationEvent(newsDetailDto, EventType.DELETED));
                itemMap.remove(newsDetailDto.getTitle());
                break;

            case null, default:
                return ResponseEntity.badRequest()
                        .body("Event type must be one of: CREATED, UPDATED, DELETED, APPROVED or REJECTED");
        }
        return ResponseEntity.ok("message sent correctly, newsDetailDto: %s".formatted(newsDetailDto));
    }
}