package com.finconsgroup.itserr.marketplace.metrics.feeder.messaging;

import com.fasterxml.jackson.core.type.TypeReference;
import com.finconsgroup.itserr.marketplace.metrics.feeder.client.MetricsBsClient;
import com.finconsgroup.itserr.marketplace.metrics.feeder.config.properties.MetricsFeederConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.favouriteuserbs.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.feeder.dto.metricbs.OutputMetricEventDto;
import feign.FeignException;
import io.cloudevents.CloudEvent;
import java.time.ZoneOffset;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

/**
 * Subscriber class that handles favorite user resource events and converts them into metric events.
 */
@Slf4j
@Component("favouriteUserEventConsumer")
public class FavouriteUserEventConsumer
        extends AbstractMetricEventConsumer<FavouriteItemMessageBodyDto> {

    private final MetricsBsClient metricsBsClient;
    private final String resourceDeletedEventType;

    /**
     * Constructor.
     *
     * @param config Configuration properties for the metrics feeder
     * @param metricsBsClient Client for interacting with the metrics business service
     */
    public FavouriteUserEventConsumer(
            final MetricsFeederConfigurationProperties config,
            final MetricsBsClient metricsBsClient) {

        super(
                metricsBsClient,
                MetricDtoType.CATALOG_ITEM_FAVOURITE,
                config.getMessaging().getFavouriteUser().getResourceCreatedEventType(),
                new TypeReference<>() {
                });

        this.metricsBsClient = metricsBsClient;
        this.resourceDeletedEventType = config.getMessaging().getFavouriteUser().getResourceDeletedEventType();

    }

    @Override
    protected void handleEvent(
            final FavouriteItemMessageBodyDto resourcePayload,
            @NonNull final CloudEvent event) {

        if (resourceDeletedEventType.equals(event.getType())) {
            handleDeleteEvent(resourcePayload, event);
        } else {
            super.handleEvent(resourcePayload, event);
        }

    }

    private void handleDeleteEvent(
            final FavouriteItemMessageBodyDto resourcePayload,
            @NonNull final CloudEvent event) {

        if (resourcePayload == null) {
            log.warn("Received {} event with no payload. Message will be dropped.", resourceDeletedEventType);
            return;
        }

        if (resourcePayload.getTimestamp() == null) {
            log.warn("Received {} event with no timestamp. Message will be dropped.", resourceDeletedEventType);
            return;
        }

        if (resourcePayload.getUser() == null || resourcePayload.getUser().getId() == null) {
            log.warn("Received {} event with no user id. Message will be dropped.", resourceDeletedEventType);
            return;
        }

        if (StringUtils.isBlank(resourcePayload.getCategory())) {
            log.warn("Received {} event with no category. Message will be dropped.", resourceDeletedEventType);
            return;
        }

        if (!StringUtils.equalsIgnoreCase(resourcePayload.getCategory(), "CATALOG")) {
            log.info("Received {} event for non-catalog category (type={}). Skipping.",
                    resourceDeletedEventType, resourcePayload.getCategory());
            return;
        }

        if (resourcePayload.getAdditionalData() == null
                || StringUtils.isBlank(resourcePayload.getAdditionalData().getItemId())) {
            log.warn("Received {} event with no item id. Message will be dropped.", resourceDeletedEventType);
            return;
        }

        final String resourceId = resourcePayload.getAdditionalData().getItemId();
        final String eventAuthor = resourcePayload.getUser().getId().toString();

        final InputCreateMetricEventDto metricEvent = InputCreateMetricEventDto.builder()
                .eventTime(resourcePayload.getTimestamp().atZone(ZoneOffset.UTC))
                .eventAuthor(eventAuthor)
                .resourceId(resourceId)
                .build();

        try {
            final OutputMetricEventDto created = metricsBsClient.createMetricEvent(
                    MetricDtoType.CATALOG_ITEM_UNFAVOURITE, metricEvent);
            log.debug("Created unfavourite metric event: {}", created.getId());
        } catch (final FeignException.FeignClientException.Conflict e) {
            log.info("Unfavourite metric event already exists for resource {} by author {}. Skipping.",
                    resourceId, eventAuthor);
        }

    }

    @Override
    protected void fillMetricEvent(
            @NonNull final InputCreateMetricEventDto.InputCreateMetricEventDtoBuilder metricEventBuilder,
            @NonNull final FavouriteItemMessageBodyDto resourcePayload,
            @NonNull final CloudEvent event) throws ValidationException {

        if (StringUtils.isBlank(resourcePayload.getCategory())) {
            throw new ValidationException("no category");
        } else if (!StringUtils.equalsIgnoreCase(resourcePayload.getCategory(), "CATALOG")) {
            throw new ValidationException("not a catalog item favourite (type=" + resourcePayload.getCategory() + ")", false);
        } else if (resourcePayload.getAdditionalData() == null) {
            throw new ValidationException("no additional data");
        } else if (StringUtils.isBlank(resourcePayload.getAdditionalData().getItemId())) {
            throw new ValidationException("no item id");
        } else {

            final String itemId = resourcePayload.getAdditionalData().getItemId();

            metricEventBuilder
                    .resourceId(itemId);

        }
    }

}
