package com.finconsgroup.itserr.marketplace.metrics.dm.service;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.InputFindDashboardMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputDashboardMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputReportMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesDailyMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesSummaryMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.ReportGranularity;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;

/**
 * Service for handling business logic related to metrics.
 */
public interface MetricService {

    /**
     * Updates metrics asynchronously.
     */
    void update();

    /**
     * Updates metrics.
     *
     * @param sync if true, the update is performed synchronously, otherwise asynchronously
     */
    void update(boolean sync);

    /**
     * Retrieves daily aggregated metrics for the specified resources within the given date range.
     *
     * @param resourcesIds the set of unique identifiers for the resources whose metrics are to be retrieved
     * @param fromDate the start date of the date range for metrics aggregation, inclusive
     * @param toDate the end date of the date range for metrics aggregation, inclusive
     * @return a {@link OutputResourcesDailyMetricsDto} containing the daily aggregated metrics for the specified resources
     */
    @NonNull
    OutputResourcesDailyMetricsDto findResourcesDailyMetrics(
            @NonNull Set<String> resourcesIds,
            @NonNull LocalDate fromDate,
            @NonNull LocalDate toDate);

    /**
     * Retrieves a summary of aggregated metrics for the specified resources.
     *
     * @param resourcesIds the set of unique identifiers for the resources whose summary metrics are to be retrieved
     * @param toDate the end date up to which metrics should be aggregated, inclusive
     * @return an instance of {@code OutputResourcesSummaryMetricsDto}, which contains the aggregated summary metrics for the specified resources
     */
    @NonNull
    OutputResourcesSummaryMetricsDto findResourcesSummaryMetrics(
            @NonNull Set<String> resourcesIds,
            @Nullable LocalDate toDate);

    /**
     * Retrieves a summary of aggregated metrics for the specified resources aggregated based on the report granularity.
     *
     * @param resourcesIds the set of unique identifiers for the resources whose summary metrics are to be retrieved
     * @param granularity  the report granularity
     * @param fromDate the start date from which metrics should be aggregated, inclusive
     * @param toDate the end date up to which metrics should be aggregated, inclusive
     * @return a list of {@code OutputReportMetricsDto},
     * which contains the aggregated summary metrics for the specified resources based on the report granularity.
     */
    @NonNull
    List<OutputReportMetricsDto> findReportingMetrics(
            @NonNull Set<String> resourcesIds,
            ReportGranularity granularity,
            LocalDate fromDate,
            LocalDate toDate);

    /**
     * Retrieves a summary of aggregated metrics across all resources aggregated based on the time resolution.
     *
     * @param inputFindDashboardMetricsDtos the dtos containing information to find dashboard metrics for
     * @return a list of {@code OutputDashboardMetricsDto},
     * which contains the aggregated summary metrics across all resources based on the time resolution.
     */
    @NonNull
    List<List<OutputDashboardMetricsDto>> findDashboardMetrics(
            @NonNull List<InputFindDashboardMetricsDto> inputFindDashboardMetricsDtos);

}
