package com.finconsgroup.itserr.marketplace.metrics.dm.service.impl;

import com.finconsgroup.itserr.marketplace.metrics.dm.bs.QueuedMVUpdater;
import com.finconsgroup.itserr.marketplace.metrics.dm.config.properties.MVUpdateConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.InputBeneficiaryDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventRefDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventBeneficiaryEntity;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventEntity;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricType;
import com.finconsgroup.itserr.marketplace.metrics.dm.exception.MetricEventConflictException;
import com.finconsgroup.itserr.marketplace.metrics.dm.exception.MetricEventNotFoundException;
import com.finconsgroup.itserr.marketplace.metrics.dm.mapper.BeneficiaryMapper;
import com.finconsgroup.itserr.marketplace.metrics.dm.mapper.MetricEventMapper;
import com.finconsgroup.itserr.marketplace.metrics.dm.mapper.MetricTypeMapper;
import com.finconsgroup.itserr.marketplace.metrics.dm.repository.MetricEventBeneficiaryRepository;
import com.finconsgroup.itserr.marketplace.metrics.dm.repository.MetricEventRepository;
import com.finconsgroup.itserr.marketplace.metrics.dm.service.MetricEventService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;

/**
 * Default implementation of {@link MetricEventService} to perform operations related to metricEvent resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetricEventService implements MetricEventService {

    private static final long UPDATE_DELAY_MS = 1000;

    private final MVUpdateConfigurationProperties config;

    private final MetricEventRepository metricEventRepository;
    private final MetricEventBeneficiaryRepository metricEventBeneficiaryRepository;

    private final MetricEventMapper metricEventMapper;
    private final MetricTypeMapper metricTypeMapper;
    private final BeneficiaryMapper beneficiaryMapper;

    private final QueuedMVUpdater queuedMVUpdater;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputMetricEventDto create(
            @NonNull final MetricDtoType metric,
            @NonNull final InputCreateMetricEventDto request) {

        // Validate metric event state
        final MetricType entityMetric = metricTypeMapper.metricDtoTypeToMetricType(metric);
        validateCreate(entityMetric, request.getResourceId(), request.getEventAuthor());

        // Save metric event
        final MetricEventEntity metricEventEntity = metricEventMapper.toEntity(metric, request);
        final MetricEventEntity savedMetricEventEntity;
        final UUID metricEventId;
        try {
            savedMetricEventEntity = metricEventRepository.saveAndFlush(metricEventEntity);
            metricEventId = savedMetricEventEntity.getId();
        } catch (final DataIntegrityViolationException e) {
            throw new MetricEventConflictException(e);
        }

        // Save metric event beneficiaries
        final List<InputBeneficiaryDto> requestBeneficiaries = request.getBeneficiaries();
        final List<MetricEventBeneficiaryEntity> savedBeneficiaries;
        if (requestBeneficiaries != null && !requestBeneficiaries.isEmpty()) {
            final List<MetricEventBeneficiaryEntity> beneficiariesEntities = requestBeneficiaries
                    .stream()
                    .map(b -> beneficiaryMapper.toEntity(b, metric, metricEventId))
                    .toList();
            savedBeneficiaries = metricEventBeneficiaryRepository.saveAllAndFlush(beneficiariesEntities);
        } else {
            savedBeneficiaries = List.of();
        }

        // Queue metrics refresh after a delay to allow the database to flush
        if (config.isOnNewEvents()) {
            queuedMVUpdater.updateAfter(UPDATE_DELAY_MS);
        }

        // Return saved metric event
        return metricEventMapper.toDto(savedMetricEventEntity, savedBeneficiaries);

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetricEventRefDto> findAll(
            @NonNull final MetricDtoType metric,
            @NonNull final Pageable pageable) {
        final MetricType entityMetric = metricTypeMapper.metricDtoTypeToMetricType(metric);
        return metricEventRepository.findAllByMetric(entityMetric, pageable)
                .map(metricEventMapper::toRefDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputMetricEventDto findById(
            @NonNull final MetricDtoType metric,
            @NonNull final UUID metricEventId) {
        final MetricType entityMetric = metricTypeMapper.metricDtoTypeToMetricType(metric);
        final List<MetricEventBeneficiaryEntity> beneficiaries = metricEventBeneficiaryRepository.findByMetricAndEventId(entityMetric, metricEventId);
        return metricEventRepository.findById(
                        MetricEventEntity.MetricEventId.of(entityMetric, metricEventId))
                .map(e -> metricEventMapper.toDto(e, beneficiaries))
                .orElseThrow(() -> new MetricEventNotFoundException(metricEventId));
    }

    /**
     * Validates that a new metric event can be created based on the current state.
     * <p>
     * Rules:
     * <ul>
     *     <li>FAVOURITE: net count (FAVOURITE - UNFAVOURITE) for the same (resourceId, eventAuthor) must be 0</li>
     *     <li>UNFAVOURITE: net count (FAVOURITE - UNFAVOURITE) for the same (resourceId, eventAuthor) must be 1</li>
     *     <li>DOWNLOAD, VIEW: count for the same (metric, resourceId, eventAuthor) must be 0</li>
     *     <li>COMMENT: no restriction</li>
     * </ul>
     */
    private void validateCreate(
            final MetricType metric,
            final String resourceId,
            final String eventAuthor) {

        if (StringUtils.isBlank(eventAuthor)) {
            return;
        }

        switch (metric) {
            case CATALOG_ITEM_FAVOURITE -> {
                final long favouriteCount = metricEventRepository.countByMetricAndResourceIdAndEventAuthor(
                        MetricType.CATALOG_ITEM_FAVOURITE, resourceId, eventAuthor);
                final long unfavouriteCount = metricEventRepository.countByMetricAndResourceIdAndEventAuthor(
                        MetricType.CATALOG_ITEM_UNFAVOURITE, resourceId, eventAuthor);
                if (favouriteCount - unfavouriteCount != 0) {
                    throw new MetricEventConflictException("Metric event already active");
                }
            }
            case CATALOG_ITEM_UNFAVOURITE -> {
                final long favouriteCount = metricEventRepository.countByMetricAndResourceIdAndEventAuthor(
                        MetricType.CATALOG_ITEM_FAVOURITE, resourceId, eventAuthor);
                final long unfavouriteCount = metricEventRepository.countByMetricAndResourceIdAndEventAuthor(
                        MetricType.CATALOG_ITEM_UNFAVOURITE, resourceId, eventAuthor);
                if (favouriteCount - unfavouriteCount != 1) {
                    throw new MetricEventConflictException("No active metric event to remove");
                }
            }
            case CATALOG_ITEM_DOWNLOAD, CATALOG_ITEM_VIEW -> {
                final long count = metricEventRepository.countByMetricAndResourceIdAndEventAuthor(
                        metric, resourceId, eventAuthor);
                if (count != 0) {
                    throw new MetricEventConflictException("Metric event already active");
                }
            }
            case CATALOG_ITEM_COMMENT -> {
                // No restriction: multiple comments per (user, resource) are allowed
            }
        }
    }

}
