package com.finconsgroup.itserr.marketplace.metrics.dm.repository;

import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventDailyCount;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricDashboardView;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;

/**
 * Repository interface for performing CRUD operations and queries on the {@link MetricEventDailyCount} entity.
 */
@Repository
public interface MetricEventDailyCountRepository extends JpaRepository<MetricEventDailyCount, MetricEventDailyCount.EntityId> {

    /**
     * Finds a list of {@link MetricEventDailyCount} entities based on the specified resource IDs and event date range.
     *
     * @param resourceIds the resource IDs to filter the results by
     * @param fromDate the start date of the range (inclusive) to filter the results by
     * @param toDate the end date of the range (inclusive) to filter the results by
     * @return a list of {@link MetricEventDailyCount} entities matching the specified resource IDs and within the given date range
     */
    List<MetricEventDailyCount> findByResourceIdInAndEventDayBetween(
            Collection<String> resourceIds,
            LocalDate fromDate,
            LocalDate toDate);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection(m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
            GROUP BY m.metric
            ORDER BY m.metric
            """)
    List<MetricSummaryProjection> summarizeByResourceIds(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection(m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                AND  m.eventDay <= :endDate
            GROUP BY m.metric
            ORDER BY m.metric
            """)
    List<MetricSummaryProjection> summarizeByResourceIdsAndEndDate(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @NonNull @Param("endDate") LocalDate endDate);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection(m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                 AND (CAST(:fromDate AS DATE) IS NULL or m.eventDay >= :fromDate)
                 AND (CAST(:toDate AS DATE) IS NULL or m.eventDay <= :toDate)
            GROUP BY m.metric
            ORDER BY m.metric
            """)
    List<MetricReportProjection> summarizeReportByGranularityUser(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection(m.resourceId, m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                 AND (CAST(:fromDate AS DATE) IS NULL or m.eventDay >= :fromDate)
                 AND (CAST(:toDate AS DATE) IS NULL or m.eventDay <= :toDate)
            GROUP BY m.resourceId, m.metric
            ORDER BY m.resourceId, m.metric
            """)
    List<MetricReportProjection> summarizeReportByGranularityResource(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection(m.eventDay, m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                 AND (CAST(:fromDate AS DATE) IS NULL or m.eventDay >= :fromDate)
                 AND (CAST(:toDate AS DATE) IS NULL or m.eventDay <= :toDate)
            GROUP BY m.eventDay, m.metric
            ORDER BY m.eventDay, m.metric
            """)
    List<MetricReportProjection> summarizeReportByGranularityDate(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);


    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection(m.resourceId, m.eventDay, m.metric, m.eventsCount)
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                 AND (CAST(:fromDate AS DATE) IS NULL or m.eventDay >= :fromDate)
                 AND (CAST(:toDate AS DATE) IS NULL or m.eventDay <= :toDate)
            ORDER BY m.resourceId, m.eventDay, m.metric
            """)
    List<MetricReportProjection> summarizeReportByGranularityResourceDate(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query(value = """
            select
                cast(:toDate as DATE) as event_day,
                NULL as date_part,
                COUNT(distinct m.resource_id) resource_count,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_DOWNLOAD' then events_count end), 0) as downloads,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_VIEW' then events_count end), 0) as views,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_FAVOURITE' then events_count end), 0) as favourites,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_COMMENT' then events_count end), 0) as comments
             from
                metric_event_daily_counts m
             where 	(cast(:fromDate as DATE) is null or m.event_day >= :fromDate)
                and (cast(:toDate as DATE) is null or m.event_day <= :toDate)
            """, nativeQuery = true)
    List<MetricDashboardView> aggregateDashboardByTimeResolutionNone(
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query(value = """
            select
                m.event_day,
                NULL as date_part,
                COUNT(distinct m.resource_id) resource_count,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_DOWNLOAD' then events_count end), 0) as downloads,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_VIEW' then events_count end), 0) as views,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_FAVOURITE' then events_count end), 0) as favourites,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_COMMENT' then events_count end), 0) as comments
             from
                metric_event_daily_counts m
             where 	(cast(:fromDate as DATE) is null or m.event_day >= :fromDate)
                and (cast(:toDate as DATE) is null or m.event_day <= :toDate)
             GROUP BY m.event_day
             ORDER BY m.event_day
            """, nativeQuery = true)
    List<MetricDashboardView> aggregateDashboardByTimeResolutionDaily(
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query(value = """
            select
                null,
                to_char(m.event_day, 'YYYY-mm') as date_part,
                COUNT(distinct m.resource_id) resource_count,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_DOWNLOAD' then events_count end), 0) as downloads,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_VIEW' then events_count end), 0) as views,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_FAVOURITE' then events_count end), 0) as favourites,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_COMMENT' then events_count end), 0) as comments
             from
                metric_event_daily_counts m
             where 	(cast(:fromDate as DATE) is null or m.event_day >= :fromDate)
                and (cast(:toDate as DATE) is null or m.event_day <= :toDate)
             GROUP BY to_char(m.event_day, 'YYYY-mm')
             ORDER BY to_char(m.event_day, 'YYYY-mm')
            """, nativeQuery = true)
    List<MetricDashboardView> aggregateDashboardByTimeResolutionMonthly(
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query(value = """
            select
                null,
                to_char(m.event_day, 'YYYY-Q') as date_part,
                COUNT(distinct m.resource_id) resource_count,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_DOWNLOAD' then events_count end), 0) as downloads,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_VIEW' then events_count end), 0) as views,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_FAVOURITE' then events_count end), 0) as favourites,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_COMMENT' then events_count end), 0) as comments
             from
                metric_event_daily_counts m
             where 	(cast(:fromDate as DATE) is null or m.event_day >= :fromDate)
                and (cast(:toDate as DATE) is null or m.event_day <= :toDate)
             GROUP BY to_char(m.event_day, 'YYYY-Q')
             ORDER BY to_char(m.event_day, 'YYYY-Q')
            """, nativeQuery = true)
    List<MetricDashboardView> aggregateDashboardByTimeResolutionQuarterly(
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

    @Query(value = """
            select
                null,
                to_char(m.event_day, 'YYYY') as date_part,
                COUNT(distinct m.resource_id) resource_count,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_DOWNLOAD' then events_count end), 0) as downloads,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_VIEW' then events_count end), 0) as views,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_FAVOURITE' then events_count end), 0) as favourites,
                coalesce(SUM(case when metric = 'CATALOG_ITEM_COMMENT' then events_count end), 0) as comments
             from
                metric_event_daily_counts m
             where 	(cast(:fromDate as DATE) is null or m.event_day >= :fromDate)
                and (cast(:toDate as DATE) is null or m.event_day <= :toDate)
             GROUP BY to_char(m.event_day, 'YYYY')
             ORDER BY to_char(m.event_day, 'YYYY')
            """, nativeQuery = true)
    List<MetricDashboardView> aggregateDashboardByTimeResolutionYearly(
            @Param("fromDate") LocalDate fromDate,
            @Param("toDate") LocalDate toDate);

}
