package com.finconsgroup.itserr.marketplace.metrics.dm.mapper;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputDashboardMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputReportMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourceMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesDailyMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesSummaryMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventDailyCount;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricTimeResolution;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricType;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricDashboardView;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricReportProjection;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection;
import org.mapstruct.Mapper;
import org.mapstruct.factory.Mappers;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.ToDoubleFunction;
import java.util.stream.Collectors;

/**
 * Mapper interface responsible for converting between MetricEvent DTOs and entities.
 */
@Mapper(config = MapperConfiguration.class, uses = { MetricTypeMapper.class })
public interface MetricMapper {

    /**
     * Converts a collection of {@link MetricEventDailyCount} entities to {@link OutputResourcesDailyMetricsDto}.
     *
     * @param entities Collection of metric event daily counts to convert
     * @return {@link OutputResourcesDailyMetricsDto} containing the converted data
     */
    default OutputResourcesDailyMetricsDto toOutputResourcesDailyMetricsDto(Collection<MetricEventDailyCount> entities) {

        if (entities == null) {
            return new OutputResourcesDailyMetricsDto();
        }

        final LinkedHashMap<LocalDate, LinkedHashMap<String, LinkedHashMap<MetricType, Long>>> grouped =
                entities.stream()
                        .filter(e -> e.getResourceId() != null && e.getMetric() != null && e.getEventDay() != null)
                        .collect(Collectors.groupingBy(
                                MetricEventDailyCount::getEventDay,
                                LinkedHashMap::new,
                                Collectors.groupingBy(
                                        MetricEventDailyCount::getResourceId,
                                        LinkedHashMap::new,
                                        Collectors.groupingBy(
                                                MetricEventDailyCount::getMetric,
                                                LinkedHashMap::new,
                                                Collectors.summingLong(e -> e.getEventsCount() == null ? 0 : e.getEventsCount())))));

        return toOutputResourcesDailyMetricsDto(grouped);

    }

    /**
     * Converts a nested map of daily metric counts to {@link OutputResourcesDailyMetricsDto}.
     *
     * @param dailyCounts Map containing daily metric counts organized by date, resource ID, and metric type
     * @return {@link OutputResourcesDailyMetricsDto} containing the converted data
     */
    default OutputResourcesDailyMetricsDto toOutputResourcesDailyMetricsDto(
            LinkedHashMap<LocalDate, LinkedHashMap<String, LinkedHashMap<MetricType, Long>>> dailyCounts) {

        final MetricTypeMapper metricTypeMapper = Mappers.getMapper(MetricTypeMapper.class);

        final OutputResourcesDailyMetricsDto result = new OutputResourcesDailyMetricsDto();
        if (dailyCounts == null) {
            return result;
        }

        for (Map.Entry<LocalDate, LinkedHashMap<String, LinkedHashMap<MetricType, Long>>> dayEntry : dailyCounts.entrySet()) {

            final LocalDate day = dayEntry.getKey();
            final LinkedHashMap<String, LinkedHashMap<MetricType, Long>> dayResources = dayEntry.getValue();
            if (dayResources == null) {
                continue;
            }

            final OutputResourcesMetricsDto resourcesMetrics = new OutputResourcesMetricsDto();
            result.put(day, resourcesMetrics);

            for (Map.Entry<String, LinkedHashMap<MetricType, Long>> resEntry : dayResources.entrySet()) {

                final String resourceId = resEntry.getKey();
                final LinkedHashMap<MetricType, Long> resourceMetrics = resEntry.getValue();
                if (resourceId == null || resourceMetrics == null) {
                    continue;
                }

                final OutputResourceMetricsDto outResourceMetrics = new OutputResourceMetricsDto();
                final Map<MetricDtoType, Double> metrics = new LinkedHashMap<>();
                outResourceMetrics.setMetrics(metrics);

                resourcesMetrics.put(resourceId, outResourceMetrics);

                for (Map.Entry<MetricType, Long> metricEntry : resourceMetrics.entrySet()) {
                    final MetricType metric = metricEntry.getKey();
                    final Long count = metricEntry.getValue();

                    if (metric == null || count == null)
                        continue;

                    final MetricDtoType metricDto;
                    try {
                        metricDto = metricTypeMapper.metricTypeToMetricDtoType(metric);
                    } catch (Exception ex) {
                        continue;
                    }

                    final Double value = count.doubleValue();
                    metrics.put(metricDto, value);

                }

            }

        }

        return result;

    }

    default OutputResourcesSummaryMetricsDto toOutputResourcesSummaryMetricsDto(
            final Collection<MetricSummaryProjection> metrics) {

        final OutputResourcesSummaryMetricsDto result = new OutputResourcesSummaryMetricsDto();

        if (metrics != null) {

            final MetricTypeMapper metricTypeMapper = Mappers.getMapper(MetricTypeMapper.class);

            metrics.forEach(m -> result.put(
                    metricTypeMapper.metricTypeToMetricDtoType(m.metricType()),
                    m.count() != null ? m.count().doubleValue() : 0.0));

        }

        return result;

    }

    default List<OutputReportMetricsDto> toOutputReportMetricsDto(
            final Collection<MetricReportProjection> metrics,
            final ToDoubleFunction<OutputResourceMetricsDto> interestMapper) {

        if (metrics == null || metrics.isEmpty()) {
            return List.of();
        }

        final List<OutputReportMetricsDto> result = new ArrayList<>(metrics.size());
        final MetricTypeMapper metricTypeMapper = Mappers.getMapper(MetricTypeMapper.class);
        String lastResourceId = null;
        LocalDate lastEventDay = null;
        OutputResourcesSummaryMetricsDto metricResult = null;
        boolean resourceIdChanged;
        boolean eventDayChanged;
        boolean isFirstIteration = true;

        for (MetricReportProjection reportMetric : metrics) {
            resourceIdChanged = lastResourceId != null && !lastResourceId.equals(reportMetric.resourceId());
            eventDayChanged = lastEventDay != null && !lastEventDay.equals(reportMetric.eventDay());

            // add to the report metric result when resource id or event day changes
            if (resourceIdChanged || eventDayChanged) {
                OutputReportMetricsDto reportResult = toOutputReportMetricsDto(metricResult, interestMapper,
                        lastResourceId, lastEventDay);
                result.add(reportResult);
            }

            if (isFirstIteration || resourceIdChanged || eventDayChanged) {
                metricResult = new OutputResourcesSummaryMetricsDto();
                lastResourceId = reportMetric.resourceId();
                lastEventDay = reportMetric.eventDay();
                isFirstIteration = false;
            }

            // add the currently processed resource id metric to metric result
            metricResult.put(
                    metricTypeMapper.metricTypeToMetricDtoType(reportMetric.metricType()),
                    reportMetric.count() != null ? reportMetric.count().doubleValue() : 0.0);
        }

        // add the final resource id
        if (metricResult != null) {
            OutputReportMetricsDto reportResult = toOutputReportMetricsDto(metricResult, interestMapper,
                    lastResourceId, lastEventDay);
            result.add(reportResult);
        }

        return result;

    }

    default OutputReportMetricsDto toOutputReportMetricsDto(final Map<MetricDtoType, Double> metrics,
                                                            final ToDoubleFunction<OutputResourceMetricsDto> interestMapper,
                                                            final String resourceId,
                                                            final LocalDate eventDay) {
        OutputResourceMetricsDto resourceResult = OutputResourceMetricsDto.builder()
                .metrics(metrics)
                .build();

        return OutputReportMetricsDto.builder()
                .resourceId(resourceId)
                .eventDay(eventDay)
                .interest(interestMapper.applyAsDouble(resourceResult))
                .downloads(metrics.getOrDefault(MetricDtoType.CATALOG_ITEM_DOWNLOAD, 0.0).longValue())
                .views(metrics.getOrDefault(MetricDtoType.CATALOG_ITEM_VIEW, 0.0).longValue())
                .favourites(metrics.getOrDefault(MetricDtoType.CATALOG_ITEM_FAVOURITE, 0.0).longValue())
                .comments(metrics.getOrDefault(MetricDtoType.CATALOG_ITEM_COMMENT, 0.0).longValue())
                .build();
    }

    default List<OutputDashboardMetricsDto> toOutputDashboardMetricsDto(
            final Collection<MetricDashboardView> dashboardMetrics,
            final MetricTimeResolution timeResolution) {

        if (dashboardMetrics == null || dashboardMetrics.isEmpty()) {
            return List.of();
        }

        final List<OutputDashboardMetricsDto> metricsDtos = new ArrayList<>(dashboardMetrics.size());

        for (MetricDashboardView dashboardMetric : dashboardMetrics) {
            var metrics = new LinkedHashMap<MetricDtoType, Double>();
            metrics.put(MetricDtoType.CATALOG_ITEM_DOWNLOAD,
                    Optional.ofNullable(dashboardMetric.getDownloads()).map(Long::doubleValue).orElse(0.0));
            metrics.put(MetricDtoType.CATALOG_ITEM_VIEW,
                    Optional.ofNullable(dashboardMetric.getViews()).map(Long::doubleValue).orElse(0.0));
            metrics.put(MetricDtoType.CATALOG_ITEM_FAVOURITE,
                    Optional.ofNullable(dashboardMetric.getFavourites()).map(Long::doubleValue).orElse(0.0));
            metrics.put(MetricDtoType.CATALOG_ITEM_COMMENT,
                    Optional.ofNullable(dashboardMetric.getComments()).map(Long::doubleValue).orElse(0.0));
            var dashboardMetricsDto = OutputDashboardMetricsDto.builder()
                    .eventDay(dashboardMetric.getEventDay())
                    .eventYear(dashboardMetric.getEventYear(timeResolution))
                    .eventMonth(dashboardMetric.getEventMonth(timeResolution))
                    .eventQuarter(dashboardMetric.getEventQuarter(timeResolution))
                    .resources(dashboardMetric.getResourceCount())
                    .metrics(metrics)
                    .build();
            metricsDtos.add(dashboardMetricsDto);
        }

        return metricsDtos;
    }

}
