package com.finconsgroup.itserr.marketplace.metrics.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.Id;
import jakarta.persistence.IdClass;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.io.Serializable;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * Entity class representing a metric event.
 */
@Entity
@Table(name = "metric_event",
        uniqueConstraints = {
                @UniqueConstraint(
                        name = "uq_metricevent_metric_resourceid_author",
                        columnNames = { "metric", "resource_id", "event_author" })
        })
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
@IdClass(MetricEventEntity.MetricEventId.class)
public class MetricEventEntity extends AbstractUUIDEntity {

    /**
     * Composite ID class for MetricEventEntity.
     */
    @Data
    @NoArgsConstructor
    @AllArgsConstructor(staticName = "of")
    public static class MetricEventId implements Serializable {

        /** The metric name that identifies the type of event */
        private MetricType metric;

        /** The UUID of the metric event */
        private UUID id;

    }

    /**
     * The name of the metric.
     */
    @Id
    @Enumerated(EnumType.STRING)
    @Column(name = "metric", nullable = false)
    private MetricType metric;

    /**
     * The id of the resource associated with the metric event.
     */
    @Column(name = "resource_id", nullable = false)
    private String resourceId;

    /** Represents the author or initiator of the metric event. */
    @Column(name = "event_author")
    private String eventAuthor;

    /**
     * The timestamp of the metric event.
     */
    @Column(name = "event_time", nullable = false)
    private ZonedDateTime eventTime;

    /**
     * The timestamp when the metricEvent was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private ZonedDateTime creationTime;

    /**
     * Sets both {@code creationTime} and {@code updateTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps. The result is truncated to microseconds to match PostgreSQL's default
     * precision for {@code TIMESTAMPTZ} columns (6 digits). Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps are immediately available
     * after {@code JpaRepository.save()}, without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        creationTime = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
    }

}
