package com.finconsgroup.itserr.marketplace.metrics.dm.bs;

import com.finconsgroup.itserr.marketplace.metrics.dm.config.MVUpdateConfiguration;
import com.finconsgroup.itserr.marketplace.metrics.dm.config.properties.MVUpdateConfigurationProperties;
import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;

/**
 * Component responsible for managing queued updates of materialized views with cooldown support.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class QueuedMVUpdater {

    /** Number of milliseconds in one minute */
    private static final long MILLIS_IN_MINUTE = 60 * 1000L;

    /** Configuration properties for materialized view updates */
    private final MVUpdateConfigurationProperties config;

    /** Publisher for materialized view update events */
    private final ApplicationEventPublisher publisher;

    /** Component that performs the actual materialized view updates */
    private final MVUpdater mvUpdater;

    /**
     * Represents the minimum timestamp in milliseconds after which the next materialized view update can be performed. This timestamp is used to enforce a
     * cooldown period between consecutive updates.
     */
    @Getter
    private long nextUpdateMinTimestampMs;

    /**
     * Event class representing a materialized view update request with timestamp.
     */
    @Data
    public static class UpdateEvent {
        private final LocalDateTime timestamp = LocalDateTime.now();
    }

    /**
     * Queues a materialized view update by publishing an update event.
     */
    public void update() {
        log.debug("Queued materialized view update");
        publisher.publishEvent(new UpdateEvent());
    }

    /**
     * Updates metrics after the specified delay.
     *
     * @param delayMs The delay in milliseconds before the update is performed
     */
    @Async
    public void updateAfter(long delayMs) {
        if (delayMs > 0) {
            try {
                Thread.sleep(delayMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        update();
    }

    /**
     * Handles the materialized view update event by performing the update and applying cooldown if configured.
     *
     * @param event The update event containing the timestamp
     * @throws InterruptedException if the cooldown sleep is interrupted
     */
    @EventListener
    @Async(MVUpdateConfiguration.MV_UPDATE_EXECUTOR)
    public void onUpdateEvent(final UpdateEvent event) throws InterruptedException {

        // Initialize
        final long now = System.currentTimeMillis();

        // Handle cooldown
        final long cooldownMs = nextUpdateMinTimestampMs - now;
        if (cooldownMs > 0) {
            log.debug("Materialized view update cooldown for {} ms", cooldownMs);
            Thread.sleep(cooldownMs);
        }

        // Update
        final boolean updated = mvUpdater.updateIfElderThan(event.timestamp);
        if (updated) {
            nextUpdateMinTimestampMs = System.currentTimeMillis() + (config.getCooldown() * MILLIS_IN_MINUTE);
        }

    }

}
