package com.finconsgroup.itserr.marketplace.metrics.dm.repository;

import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventDailyCount;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;

/**
 * Repository interface for performing CRUD operations and queries on the {@link MetricEventDailyCount} entity.
 */
@Repository
public interface MetricEventDailyCountRepository extends JpaRepository<MetricEventDailyCount, MetricEventDailyCount.EntityId> {

    /**
     * Finds a list of {@link MetricEventDailyCount} entities based on the specified resource IDs and event date range.
     *
     * @param resourceIds the resource IDs to filter the results by
     * @param fromDate the start date of the range (inclusive) to filter the results by
     * @param toDate the end date of the range (inclusive) to filter the results by
     * @return a list of {@link MetricEventDailyCount} entities matching the specified resource IDs and within the given date range
     */
    List<MetricEventDailyCount> findByResourceIdInAndEventDayBetween(
            Collection<String> resourceIds,
            LocalDate fromDate,
            LocalDate toDate);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection(m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
            GROUP BY m.metric
            ORDER BY m.metric
            """)
    List<MetricSummaryProjection> summarizeByResourceIds(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds);

    @Query("""
            SELECT   new com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection(m.metric, SUM(m.eventsCount))
            FROM     MetricEventDailyCount m
            WHERE    m.resourceId IN :resourcesIds
                AND  m.eventDay <= :endDate
            GROUP BY m.metric
            ORDER BY m.metric
            """)
    List<MetricSummaryProjection> summarizeByResourceIdsAndEndDate(
            @NonNull @Param("resourcesIds") Collection<String> resourcesIds,
            @NonNull @Param("endDate") LocalDate endDate);

}
