package com.finconsgroup.itserr.marketplace.metrics.dm.api;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventRefDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to MetricEvents Management.
 */
@Tag(
        name = "MetricEvent",
        description = "The MetricEvent API: it provides endpoints for managing metric events."
)
@SecurityRequirement(name = "BearerAuth")
public interface MetricEventApi {

    /**
     * Creates a new metricEvent.
     *
     * @param metric the metric type
     * @param inputCreateMetricEventDto the input data transfer object containing metricEvent details
     * @return the created {@link OutputMetricEventDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create metricEvent",
            responses = { @ApiResponse(responseCode = "201", description = "Created") }
    )
    @PostMapping(
            value = "/metrics/metrics/{metricId}/events",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputMetricEventDto createMetricEvent(
            @PathVariable("metricId") MetricDtoType metric,
            @Valid @RequestBody InputCreateMetricEventDto inputCreateMetricEventDto);

    /**
     * Retrieves a paginated list of all MetricEvents.
     *
     * @param metric the metric type
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize the number of MetricEvents per page (default is 10)
     * @param sort the field to sort by (default is "id")
     * @param direction the direction of sorting (default is ascending)
     * @return a page of {@link OutputMetricEventRefDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all MetricEvents",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(value = "/metrics/metrics/{metricId}/events", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputMetricEventRefDto> findAll(
            @PathVariable("metricId") MetricDtoType metric,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a metricEvent by id.
     *
     * @param metric the metric type
     * @param metricEventId the id of the metricEvent to retrieve
     * @return the found {@link OutputMetricEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find metricEvent by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(value = "/metrics/metrics/{metricId}/events/{metricEventId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputMetricEventDto findById(
            @PathVariable("metricId") MetricDtoType metric,
            @PathVariable("metricEventId") UUID metricEventId);

}
