package com.finconsgroup.itserr.marketplace.metrics.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ExecutionException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.metrics.bs.bean.MetricsReportRequest;
import com.finconsgroup.itserr.marketplace.metrics.bs.client.MetricDmClient;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ExportFileFormat;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputCatalogItemMinimalDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsComparisonDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsInfoAtDateDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsInfoAtTodayDateDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsInterestsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ReportGranularity;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputReportMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputUserInterestInfoDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.repository.CatalogRepository;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricInterestService;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricReportService;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link MetricService} to perform operations related to user-related metrics.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetricService implements MetricService {

    /** Client for interacting with the Catalog BS service. */
    private final CatalogRepository catalogRepository;
    /** Client for interacting with the Metrics DM service. */
    private final MetricDmClient metricDmClient;
    /** Service for handling report related operations */
    private final MetricReportService metricReportService;

    /** Service for retrieving metric interest score. */
    private final MetricInterestService metricInterestService;

    // TODO: very complex logic should be split in different components
    @Override
    public OutputMetricsComparisonDto getUserMetrics(final LocalDate comparisonDate) {

        // check user is logged in and get userId
        final UUID userId = JwtTokenHolder.getUserIdOrThrow();

        // get the user catalog item ids from catalog service
        final Set<String> itemsIds = catalogRepository.getUserItemIds(userId);

        // use the resources to get the aggregated metrics from metrics-dm
        final OutputMetricsDto metrics = new OutputMetricsDto();
        final OutputMetricsDto cmqMetrics = new OutputMetricsDto();
        if (!itemsIds.isEmpty()) {
            metrics.putAll(metricDmClient.findResourcesSummaryMetrics(itemsIds, null));
            if (comparisonDate != null) {
                cmqMetrics.putAll(metricDmClient.findResourcesSummaryMetrics(itemsIds, comparisonDate));
            }
        }

        // calculate interest score
        final OutputMetricsInterestsDto todayMetricsInterests = metricInterestService.getUserMetricsInterests(metrics);
        final OutputMetricsInterestsDto cmpMetricsInterests = metricInterestService.getUserMetricsInterests(cmqMetrics);
        final double todayInterestScore = metricInterestService.getTotalInterest(todayMetricsInterests);
        final double cmpInterestScore = metricInterestService.getTotalInterest(cmpMetricsInterests);

        // compare with other users
        final OutputUserInterestInfoDto interestInfo = metricDmClient.getUserInterestInfo(userId);
        final double interestWorseUsersRatio = (double) interestInfo.getWorseEventsBeneficiaries() / (double) interestInfo.getTotalEventsBeneficiaries();
        final double roundedInterestWorseUsersRatio = Math.round(interestWorseUsersRatio * 100.0) / 100.0;

        // build result
        return OutputMetricsComparisonDto.builder()
                .today(OutputMetricsInfoAtTodayDateDto.builder()
                        .metrics(metrics)
                        .interest(todayInterestScore)
                        .metricsInterest(todayMetricsInterests)
                        .interestWorseUsersRatio(roundedInterestWorseUsersRatio)
                        .build())
                .cmpDate(
                        comparisonDate != null
                                ? OutputMetricsInfoAtDateDto.builder()
                                .date(comparisonDate)
                                .metrics(cmqMetrics)
                                .interest(cmpInterestScore)
                                .metricsInterest(cmpMetricsInterests)
                                .build()
                                : null)
                .build();

    }

    @Override
    @NonNull
    public InputStreamResource downloadReport(@NonNull final ExportFileFormat format,
                                              final LocalDate fromDate,
                                              final LocalDate toDate,
                                              final Locale locale) {
        // check user is logged in and get userId
        final UUID userId = JwtTokenHolder.getUserIdOrThrow();

        // get the user catalog items from catalog service
        final List<OutputCatalogItemMinimalDto> items = catalogRepository.getUserItems(userId);
        final Map<String, OutputCatalogItemMinimalDto> itemById = items.stream().collect(Collectors.toMap(
                OutputCatalogItemMinimalDto::getId, Function.identity()
        ));
        final Set<String> itemsIds = itemById.keySet();

        try {

            List<OutputReportMetricsDto> summaryMetrics = List.of();
            List<OutputReportMetricsDto> resourcesMetrics = List.of();

            if (!itemsIds.isEmpty()) {
                summaryMetrics = metricDmClient.findReportingMetrics(itemsIds, ReportGranularity.USER, fromDate, toDate);
                resourcesMetrics = metricDmClient.findReportingMetrics(itemsIds, ReportGranularity.RESOURCE, fromDate, toDate);
                resourcesMetrics.forEach(reportMetricsDto ->
                    reportMetricsDto.setResourceTitle(itemById.get(reportMetricsDto.getResourceId()).getTitle())
                );
            }

            MetricsReportRequest metricsReportRequest = MetricsReportRequest.builder()
                    .summaryMetrics(summaryMetrics)
                    .resourcesMetrics(resourcesMetrics)
                    .locale(locale)
                    .build();

            return metricReportService.downloadReport(format, metricsReportRequest);
        } catch (Exception ex) {
            throw new WP2ExecutionException(ex);
        }
    }

}
