package com.finconsgroup.itserr.marketplace.metrics.bs.service.impl;

import com.finconsgroup.itserr.marketplace.metrics.bs.client.MetricDmClient;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsInterestsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputMetricsInterestWeightsConfigurationDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricInterestService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.Optional;

/**
 * Default implementation of {@link MetricInterestService} to perform operations related to user-related metrics interest.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetricInterestService implements MetricInterestService {

    /** Interest weights timeout in seconds. */
    private static final long INTEREST_WEIGHTS_TIMEOUT_SECS = 60 * 5;

    /** Client for interacting with the Metrics DM service. */
    private final MetricDmClient metricDmClient;

    /** Cached interest weights. */
    private OutputMetricsInterestWeightsConfigurationDto interestWeights;
    /** Timestamp of interest weights timeout. */
    private Long interestWeightTimeout;

    @Override
    public OutputMetricsInterestsDto getUserMetricsInterests(final OutputMetricsDto metrics) {

        // Retrieve metrics interest weights if missing or timedout
        synchronized (this) {
            if (interestWeights == null || interestWeightTimeout == null || interestWeightTimeout < System.currentTimeMillis()) {
                final OutputMetricsInterestWeightsConfigurationDto actualInterestWeights =
                        Optional.ofNullable(metricDmClient.getInterestWeights())
                                .orElseGet(OutputMetricsInterestWeightsConfigurationDto::new);
                actualInterestWeights.entrySet().removeIf(entry -> entry.getValue() == null);
                interestWeights = actualInterestWeights;
                interestWeightTimeout = System.currentTimeMillis() + (INTEREST_WEIGHTS_TIMEOUT_SECS * 1000);
            }
        }

        // Calculate interest scores for each metric
        final OutputMetricsInterestsDto metricsInterests = new OutputMetricsInterestsDto();
        metrics.forEach((metric, value) -> metricsInterests.put(metric, value * interestWeights.getOrDefault(metric, 0.0)));

        // Return interest scores
        return metricsInterests;

    }

    @Override
    public double getTotalInterest(OutputMetricsInterestsDto metrics) {
        return metrics != null
                ? metrics.values()
                .stream()
                .filter(Objects::nonNull)
                .mapToDouble(e -> e)
                .sum()
                : 0;
    }
}
