package com.finconsgroup.itserr.marketplace.metrics.bs.controller;

import com.finconsgroup.itserr.marketplace.metrics.bs.api.UserMetricApi;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ExportFileFormat;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsComparisonDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.service.MetricService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDate;
import java.util.Locale;

/**
 * REST controller implementation of {@link UserMetricApi}.
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class UserMetricController implements UserMetricApi {

    private static final String DOWNLOAD_FILE_NAME_FORMAT = "attachment; filename=\"metricsReport.%s\"";

    /** Service for managing and retrieving user-related metrics. */
    private final MetricService metricService;

    @Override
    public OutputMetricsComparisonDto getUserMetrics(final LocalDate comparisonDate) {
        return metricService.getUserMetrics(comparisonDate);
    }

    public ResponseEntity<Resource> downloadUserMetrics(@NonNull final ExportFileFormat format,
                                                        final LocalDate fromDate,
                                                        final LocalDate toDate,
                                                        final Locale locale) {
        // map format to file extension
        String extension = switch (format) {
            case CSV -> "csv";
            case PDF -> "pdf";
            default -> "xlsx";
        };

        // map format to content type response header
        MediaType contentType = switch (format) {
            case CSV -> MediaType.valueOf("text/csv");
            case PDF -> MediaType.APPLICATION_PDF;
            default -> MediaType.valueOf("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        };

        InputStreamResource resource = metricService.downloadReport(format, fromDate, toDate, locale);

        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, DOWNLOAD_FILE_NAME_FORMAT.formatted(extension))
                .contentType(contentType)
                .body(resource);
    }

}
