package com.finconsgroup.itserr.marketplace.metrics.bs.be.validate;

import com.finconsgroup.itserr.marketplace.catalog.bs.dto.OutputItemDto;
import com.finconsgroup.itserr.marketplace.catalog.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.InputBeneficiaryDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.bs.exception.MetricEventValidationException;
import com.finconsgroup.itserr.marketplace.metrics.bs.repository.CatalogRepository;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Validator for catalog-related metric events. This class validates metrics related to catalog items, ensuring that the event author is not the catalog item's
 * author or maintainer, and that the provided resource IDs and author information are valid.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class CatalogMetricEventValidator implements MetricEventValidator {

    private final CatalogRepository catalogRepository;

    @Override
    public void validate(
            @NonNull final MetricDtoType metric,
            @NonNull final InputCreateMetricEventDto request) {

        // if not a catalog metric, nothing to validate
        if (!isCatalogMetric(metric)) {
            return;
        }

        // Get the catalog item id
        final UUID catalogItemId = resolveCatalogItemId(request.getResourceId());

        // Check author is present
        final UUID eventAuthor = request.getEventAuthor();
        if (eventAuthor == null) {
            throw new MetricEventValidationException("Event author is required for catalog items events");
        }

        // Get catalog item
        final OutputItemDto catalogItem = catalogRepository.findById(catalogItemId)
                .orElseThrow(() -> new MetricEventValidationException("Catalog item not found"));

        // Get catalog item maintainer and authors
        final Set<String> maintainerAndAuthors = getMaintainerAndAuthors(catalogItem);

        // Check that not catalog item author or maintainer
        if (maintainerAndAuthors.contains(eventAuthor.toString().toLowerCase())) {
            throw new MetricEventValidationException("Event author cannot be the same as the item maintainer or author");
        }

        // Include maintainer and authors as beneficiaries
        // TODO: enrichment activity should be moved to other components
        final List<InputBeneficiaryDto> requestBeneficiaries = request.getBeneficiaries() != null
                ? new ArrayList<>(request.getBeneficiaries())
                : new ArrayList<>();
        final Set<String> existingBeneficiaries =
                requestBeneficiaries.stream()
                        .map(InputBeneficiaryDto::getUser)
                        .filter(StringUtils::hasText)
                        .map(String::trim)
                        .map(String::toLowerCase)
                        .collect(Collectors.toSet());
        final Set<String> missingBeneficiaries = maintainerAndAuthors.stream()
                .filter(id -> !existingBeneficiaries.contains(id))
                .collect(Collectors.toSet());
        missingBeneficiaries.stream()
                .map(u -> InputBeneficiaryDto.builder()
                        .user(u)
                        .build())
                .forEach(requestBeneficiaries::add);
        request.setBeneficiaries(requestBeneficiaries);

    }

    /**
     * Determines if the given metric type is related to catalog operations.
     *
     * @param metric the metric type to check
     * @return true if the metric is catalog-related, false otherwise
     */
    private static boolean isCatalogMetric(MetricDtoType metric) {
        return switch (metric) {
            case CATALOG_ITEM_VIEW, CATALOG_ITEM_DOWNLOAD, CATALOG_ITEM_COMMENT, CATALOG_ITEM_FAVOURITE -> true;
            default -> false;
        };
    }

    /**
     * Resolves and validates a catalog item ID from the provided resource ID string.
     *
     * @param resourceId the resource ID to resolve
     * @return the resolved UUID
     * @throws MetricEventValidationException if the resource ID is blank or not a valid UUID
     */
    @NonNull
    private UUID resolveCatalogItemId(
            final String resourceId) {

        // Check that resource id is not blank
        if (!StringUtils.hasText(resourceId)) {
            throw new MetricEventValidationException("Resource id is required");
        }

        // Parse UUID
        try {
            return UUID.fromString(resourceId);
        } catch (final Exception e) {
            throw new MetricEventValidationException("Invalid catalog item id format (not an UUID)");
        }

    }

    /**
     * Validates that the event author is not the catalog item's author or maintainer.
     *
     * @param eventAuthor the email of the event author
     * @param item the catalog item
     * @throws MetricEventValidationException if the event author is the item's author or maintainer
     */
    private void validateNotCatalogItemAuthor(
            @NonNull final UUID eventAuthor,
            @NonNull final OutputItemDto item) {

        final Set<String> maintainerAndAuthorsIds = getMaintainerAndAuthors(item);
        final boolean isCatalogItemMaintainerOrAuthor =
                maintainerAndAuthorsIds.contains(eventAuthor.toString().toLowerCase());

        if (isCatalogItemMaintainerOrAuthor) {
            throw new MetricEventValidationException("Event author cannot be the same as the item maintainer or author");
        }
    }

    private static @NonNull Set<String> getMaintainerAndAuthors(@NonNull final OutputItemDto item) {

        final Set<String> maintainerAndAuthorsIds = new HashSet<>();

        Optional.ofNullable(item.getMaintainer())
                .map(OutputUserProfileDto::getId)
                .map(UUID::toString)
                .map(String::toLowerCase)
                .ifPresent(maintainerAndAuthorsIds::add);

        if (item.getAuthors() != null) {
            item.getAuthors().stream()
                    .filter(Objects::nonNull)
                    .map(a -> a.getId() != null ? a.getId().toString() : a.getEmail())
                    .filter(StringUtils::hasText)
                    .map(String::trim)
                    .map(String::toLowerCase)
                    .forEach(maintainerAndAuthorsIds::add);
        }

        return maintainerAndAuthorsIds;

    }

    private UUID toUUIDNoThrow(final String s) {
        if (s == null) {
            return null;
        }
        try {
            return UUID.fromString(s);
        } catch (Exception e) {
            return null;
        }
    }

}
