/**
 * Global exception handler for REST controllers in the realestate system. This class handles custom exceptions related to saleDeed and category operations and
 * returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a SaleDeedNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.metrics.bs.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.UNPROCESSABLE_ENTITY;

/**
 * Custom Exception Handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles {@link MetricEventValidationException} and returns a 422 response.
     *
     * @param ex the exception that was thrown during metric event validation
     * @param request the web request during which the exception occurred
     * @return a {@link ResponseEntity} containing an {@link ErrorResponseDto} with error details and an HTTP status of 422 (Unprocessable Entity)
     */
    @ExceptionHandler(MetricEventValidationException.class)
    public ResponseEntity<ErrorResponseDto> handleMetricEventValidationException(
            final MetricEventValidationException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(UNPROCESSABLE_ENTITY, ex.getMessage());
    }

}
