package com.finconsgroup.itserr.marketplace.metrics.bs.dto;

import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;

/**
 * Enum representing different types of UX metrics that can be tracked.
 */
@Getter
@Schema(name = "OutputUXMetricEventType", description = "Type of metrics")
public enum UXMetricDtoType {

    /** Tracks when a user downloads a catalog item */
    @JsonProperty("catalog-item-download")
    CATALOG_ITEM_DOWNLOAD,

    /** Tracks when a user views a catalog item */
    @JsonProperty("catalog-item-view")
    CATALOG_ITEM_VIEW;

    /** The string representation of the metric type used in view layer */
    private final String viewValue;

    /**
     * Default constructor that initializes viewValue from JsonProperty annotation.
     *
     * @throws IllegalStateException if enum field cannot be instantiated
     */
    UXMetricDtoType() {
        try {
            this.viewValue = this.getClass()
                    .getField(this.name())
                    .getAnnotation(JsonProperty.class)
                    .value();
        } catch (final Exception e) {
            // Cannot happen
            throw new IllegalStateException("Cannot instantiate enum", e);
        }
    }

    /**
     * Converts a view value string to corresponding MetricDtoType enum.
     *
     * @param value the string value to convert
     * @return the matching MetricDtoType, or null if value is null
     * @throws IllegalArgumentException if no matching enum constant is found
     */
    public static UXMetricDtoType fromViewValue(final String value) {
        if (value == null) {
            return null;
        }
        for (UXMetricDtoType t : values()) {
            if (t.viewValue.equals(value))
                return t;
        }
        throw new IllegalArgumentException("Invalid metric view value: " + value);
    }

}
