package com.finconsgroup.itserr.marketplace.metrics.bs.client;

import com.finconsgroup.itserr.marketplace.metrics.bs.dto.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputReportMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ReportGranularity;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputMetricsInterestWeightsConfigurationDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputResourcesSummaryMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.metrics.OutputUserInterestInfoDto;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * This interface defines the contract for interacting with metrics-dm microservice.
 */
public interface MetricDmClient {

    /**
     * Creates a new metricEvent.
     *
     * @param metric the metric type
     * @param inputCreateMetricEventDto the input data transfer object containing metricEvent details
     * @return the created {@link OutputMetricEventDto} and HTTP status 201 (Created)
     */
    @PostMapping(
            value = "/metrics/metrics/{metricId}/events",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputMetricEventDto createMetricEvent(
            @PathVariable("metricId") String metric,
            @Valid @RequestBody InputCreateMetricEventDto inputCreateMetricEventDto);

    /**
     * Summarizes the total metrics of the specified resources, combining their metrics into a single summary.
     *
     * @param resourcesIds the set of resource IDs for which the metrics summary is to be retrieved; must not be empty
     * @param toDate the optional end date for calculating the metrics summary; if null, the current date is assumed
     * @return an {@link OutputResourcesSummaryMetricsDto} containing the aggregated metrics for the specified resources
     */
    @GetMapping(value = "/metrics/metrics-summaries", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputResourcesSummaryMetricsDto findResourcesSummaryMetrics(
            @RequestParam("resourceId") @NotEmpty Set<String> resourcesIds,
            @RequestParam(value = "toDate", required = false) LocalDate toDate);

    /**
     * Retrieves the metrics interest weights configuration.
     *
     * @return an {@link OutputMetricsInterestWeightsConfigurationDto} containing the configuration of metrics interest weights as a mapping of metric
     *         identifiers to weights
     */
    @GetMapping(value = "/metrics/config/metrics-weights", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputMetricsInterestWeightsConfigurationDto getInterestWeights();

    /**
     * Retrieves interest information for a user.
     */
    @GetMapping(value = "/metrics/users/{userId}/interest", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputUserInterestInfoDto getUserInterestInfo(
            @PathVariable("userId") UUID userId);

    /**
     * Summarizes the total metrics of the specified resources, combining their metrics into a single summary based
     * on the report granularity.
     *
     * @param resourcesIds the set of resource IDs for which the metrics summary is to be retrieved; must not be empty
     * @param granularity  the report granularity
     * @param fromDate the start date of the date range (inclusive)
     * @param toDate the end date of the date range (inclusive)
     * @return a list of {@link OutputReportMetricsDto} containing the aggregated metrics for the specified resources
     */
    @GetMapping(value = "/metrics/report/metrics-summaries", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputReportMetricsDto> findReportingMetrics(
            @RequestParam("resourceId") Set<String> resourcesIds,
            @RequestParam("granularity") ReportGranularity granularity,
            @RequestParam(value = "fromDate") LocalDate fromDate,
            @RequestParam(value = "toDate") LocalDate toDate);

}
