package com.finconsgroup.itserr.marketplace.metrics.bs.api;

import com.finconsgroup.itserr.marketplace.metrics.bs.dto.ExportFileFormat;
import com.finconsgroup.itserr.marketplace.metrics.bs.dto.OutputMetricsComparisonDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.constraints.NotNull;
import org.springframework.core.io.Resource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;

import java.time.LocalDate;
import java.util.Locale;

/**
 * API interface defining endpoints for retrieving user-related metrics.
 */
@Tag(
        name = "UserMetricEvent",
        description = "The UserMetricEvent API: it provides endpoints for retrieving user-related metrics."
)
@SecurityRequirement(name = "BearerAuth")
public interface UserMetricApi {

    /**
     * Retrieves metrics for the current user.
     *
     * @param comparisonDate optional date to get a second set of metrics up to the given date for comparison reasons
     * @return list of user metrics
     */
    @Operation(
            summary = "get user metrics",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(
            value = "/metrics/me/metrics",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    OutputMetricsComparisonDto getUserMetrics(
            @Parameter(
                    description = "Optional date for comparison metrics (format: yyyy-MM-dd)",
                    schema = @Schema(type = "string", format = "date"),
                    example = "2025-11-07"
            )
            @RequestParam(name = "cmpDate", required = false) LocalDate comparisonDate);

    /**
     * Downloads metrics report for the current user.
     *
     * @param format  the format in which the file should be exported
     * @param fromDate the start date of the date range (inclusive)
     * @param toDate the end date of the date range (inclusive)
     * @return resource to download
     */
    @Operation(
            summary = "download user metrics",
            responses = {@ApiResponse(responseCode = "200", description = "OK")},
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, name = "Accept-Language", example = "en")
            }
    )
    @GetMapping(
            value = "/metrics/me/metrics/download"
    )
    ResponseEntity<Resource> downloadUserMetrics(
            @Parameter(
                    description = "Format in which the report should be exported",
                    example = "EXCEL"
            )
            @NotNull @RequestParam(name = "format") ExportFileFormat format,
            @Schema(example = "2025-08-08")
            @RequestParam(value = "fromDate", required = false) LocalDate fromDate,
            @Schema(example = "2025-08-10")
            @RequestParam(value = "toDate", required = false) LocalDate toDate,
            @Schema(hidden = true) Locale locale);

}
