package com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;
import org.jetbrains.annotations.NotNull;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

/**
 * Represents the available metadata types in the marketplace.
 * <p>
 * Supported types include:
 * <ul>
 *     <li>{@link #STANDARD} — Standard metadata.</li>
 *     <li>{@link #PERSONAL} — Personal metadata.</li>
 *     <li>{@link #COMMUNITY} — Community-shared metadata.</li>
 * </ul>
 */
@Getter
@Schema(description = "Enumeration representing the possible categories of metadata.")
public enum MetadataCategoryEnum {

    /**
     * Standard metadata type.
     */
    STANDARD("standard"),

    /**
     * Personal metadata type.
     */
    PERSONAL("personal"),

    /**
     * Community metadata type.
     */
    COMMUNITY("community");

    /**
     * The string value representing the metadata type.
     */
    private final String value;

    MetadataCategoryEnum(String value) {
        this.value = value;
    }

    @JsonValue
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    @JsonCreator
    public static MetadataCategoryEnum fromString(String value) {
        if (value == null) {
            return null;
        }
        for (MetadataCategoryEnum status : MetadataCategoryEnum.values()) {
            if (status.value.equalsIgnoreCase(value)) {
                return status;
            }
        }
        throw new IllegalArgumentException("Unknown MetadataCategoryEnum: " + value);
    }

    @Component
    public static class MetadataCategoryEnumConverter implements Converter<String, MetadataCategoryEnum> {
        @Override
        public MetadataCategoryEnum convert(@NotNull String source) {
            return MetadataCategoryEnum.fromString(source);
        }
    }
}