package com.finconsgroup.itserr.marketplace.metadata.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.SecurityRoles;
import com.finconsgroup.itserr.marketplace.metadata.dm.authorization.HasRoles;
import com.finconsgroup.itserr.marketplace.metadata.dm.authorization.RequireMetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.ModerationResult;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.ModerationStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.MetadataRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification.MetadataSpecifications;
import com.finconsgroup.itserr.marketplace.metadata.dm.service.MetadataModerationService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Set;
import java.util.UUID;

/**
 * Default implementation of ItemModerationService.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetadataModerationService implements MetadataModerationService {

    private final MetadataRepository metadataRepository;
    private final MetadataMapper metadataMapper;

    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @HasRoles(SecurityRoles.MEMBER_ROLE)
    public OutputPageDto<OutputMetadataDto> getMetadataListForModeration(ModerationStatus status, String searchText, UUID userId, Pageable pageable) {
        Specification<MetadataEntity> spec;

        if (status == null) {
            Specification<MetadataEntity> nonStatuses =
                    MetadataSpecifications.hasStatusNotIn(
                            Set.of(MetadataStatus.APPROVED, MetadataStatus.REJECTED, MetadataStatus.DRAFT)
                    );

            Specification<MetadataEntity> creatorStatuses =
                    MetadataSpecifications.hasStatusIn(
                            Set.of(MetadataStatus.APPROVED, MetadataStatus.REJECTED)
                    ).and(MetadataSpecifications.updatedBy(userId));

            spec = nonStatuses.or(creatorStatuses);
        } else if (status == ModerationStatus.PENDING) {
            spec = MetadataSpecifications.hasStatusIn(Set.of(MetadataStatus.PENDING));
        } else {
            spec = MetadataSpecifications.hasStatusIn(Set.of(status.toMetadataStatus()))
                    .and(MetadataSpecifications.updatedBy(userId));
        }

        // Apply search text filter (name + description)
        spec = spec.and(MetadataSpecifications.hasSearchText(searchText));

        return OutputPageDto.fromPage(
                metadataRepository.findAll(spec, pageable)
                        .map(metadataMapper::metadataEntityToMetadataDto)
        );
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @HasRoles(SecurityRoles.MEMBER_ROLE)
    @RequireMetadataStatus(allowed = {MetadataStatus.PENDING, MetadataStatus.REJECTED, MetadataStatus.APPROVED})
    public OutputMetadataDto getMetadataForModeration(UUID metadataId) {
        // Fetch the Metadata
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Metadata not found: %s", metadataId)));
        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @HasRoles(SecurityRoles.MEMBER_ROLE)
    @RequireMetadataStatus(allowed = {MetadataStatus.PENDING})
    public OutputMetadataDto setModerationStatus(UUID metadataId, InputModerationStatusDto inputModerationStatusDto, UUID userId) {
        // Fetch the Metadata
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Metadata not found: %s", metadataId)));

        // Set the moderation message
        metadataEntity.setModerationMessage(inputModerationStatusDto.getMessage());

        // Set updatedBy
        metadataEntity.setUpdatedBy(userId);

        // Handle approval or rejection
        ModerationResult status = inputModerationStatusDto.getStatus();
        if (status == ModerationResult.APPROVED) {
            // Set status to APPROVED
            metadataEntity.setStatus(MetadataStatus.APPROVED);
            log.info("Metadata {} approved by moderator {}", metadataId, userId);
        } else if (status == ModerationResult.REJECTED) {
            // Set status to REJECTED
            metadataEntity.setStatus(MetadataStatus.REJECTED);
            log.info("Metadata {} rejected by moderator {}", metadataId, userId);
        }

        // Save the updated item
        metadataEntity = metadataRepository.saveAndFlush(metadataEntity);

        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }
}
