package com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification;

import com.finconsgroup.itserr.marketplace.metadata.dm.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import jakarta.persistence.criteria.CriteriaBuilder;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * {@link MetadataEntity} specifications.
 */
public final class MetadataSpecifications {

    /**
     * Private constructor.
     */
    private MetadataSpecifications() {
        throw new UnsupportedOperationException("Cannot be instantiated");
    }

    /**
     * <p>Creates a {@link Specification} to find {@link MetadataEntity} by argument ids.</p>
     * <p>If null or empty list is passed, a {@link CriteriaBuilder#conjunction()} is returned, matching any entity.</p>
     *
     * @param ids Metadata ids, or null.
     * @return A {@link Specification} matching ids.
     */
    @NonNull
    public static Specification<MetadataEntity> ids(@Nullable final Collection<UUID> ids) {
        return (root, query, cb) ->
                sanitizeIds(ids)
                        // TODO: may we use JPA Metamodel?
                        .map(root.get("id")::in)
                        .orElse(cb.conjunction());
    }

    /**
     * <p>Given a collection of ids, returns an {@link Optional} containing a {@link Set} with unique non-null ids.</p>
     * <p>If the argument is null or the resulting Set is empty, an empty Optional is returned.</p>
     *
     * @param ids IDs. May be null.
     * @return Optional containing a Set of unique non-null ids, or empty Optional if the resulting set is empty.
     */
    private static Optional<Set<UUID>> sanitizeIds(@Nullable final Collection<UUID> ids) {

        // If ids is null, return empty optional
        if (ids == null) {
            return Optional.empty();
        }

        // Convert to Set, excluding null values
        final Set<UUID> sanitizedIds = ids.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // Return an Optional with the non-empty Set, or an empty Optional if Set is empty
        return sanitizedIds.isEmpty()
                ? Optional.empty()
                : Optional.of(sanitizedIds);

    }

    /**
     * Creates a {@link Specification} to find {@link MetadataEntity} by category.
     *
     * @param category Metadata category, or null.
     * @return A {@link Specification} matching category.
     */
    @NonNull
    public static Specification<MetadataEntity> hasCategory(
            MetadataCategoryEnum category) {
        return (root, query, cb) ->
                category == null
                        ? cb.conjunction()
                        : cb.equal(root.get("category"), category);
    }

    /**
     * Creates a {@link Specification} to find {@link MetadataEntity} by status.
     *
     * @param statuses Metadata statuses, or null.
     * @return A {@link Specification} matching statuses.
     */
    @NonNull
    public static Specification<MetadataEntity> hasStatusIn(
            Set<MetadataStatus> statuses) {
        return (root, query, cb) ->
                statuses == null || statuses.isEmpty()
                        ? cb.conjunction()
                        : root.get("status").in(statuses);
    }

    /**
     * Creates a {@link Specification} to find {@link MetadataEntity} excluding the given statuses.
     *
     * @param statuses Metadata statuses to exclude, or null.
     * @return A {@link Specification} excluding the given statuses.
     */
    @NonNull
    public static Specification<MetadataEntity> hasStatusNotIn(Set<MetadataStatus> statuses) {
        return (root, query, cb) ->
                statuses == null || statuses.isEmpty()
                        ? cb.conjunction()
                        : cb.not(root.get("status").in(statuses));
    }

    /**
     * Creates a {@link Specification} to find {@link MetadataEntity} by creatorId.
     *
     * @param creatorId Metadata creatorId, or null.
     * @return A {@link Specification} matching creatorId.
     */
    @NonNull
    public static Specification<MetadataEntity> hasCreatorId(
            UUID creatorId) {
        return (root, query, cb) ->
                creatorId == null
                        ? cb.conjunction()
                        : cb.equal(root.get("creatorId"), creatorId);
    }

    /**
     * Creates a {@link Specification} to find {@link MetadataEntity} by updatedBy.
     *
     * @param updatedBy Metadata updated by, or null.
     * @return A {@link Specification} matching updatedBy.
     */
    @NonNull
    public static Specification<MetadataEntity> updatedBy(
            UUID updatedBy) {
        return (root, query, cb) ->
                updatedBy == null
                        ? cb.conjunction()
                        : cb.equal(root.get("updatedBy"), updatedBy);
    }


    /**
     * Creates a {@link Specification} to exclude {@link MetadataEntity} by category.
     *
     * @param category Metadata category to exclude.
     * @return A {@link Specification} excluding category.
     */
    @NonNull
    public static Specification<MetadataEntity> excludeCategory(MetadataCategoryEnum category) {
        return (root, query, cb) ->
                cb.notEqual(root.get("category"), category);
    }

    /**
     *  Creates a {@link Specification} to find {@link MetadataEntity} by name containing the given value, case-insensitive.
     * 
     *  @param namePart The value that the name should contain, case-insensitive. If null, no filtering is applied.
     *  @return A {@link Specification} matching name containing the given value, or a specification that matches all if namePart is null.
     * */
     @NonNull
     public static Specification<MetadataEntity> nameContainsIgnoreCase(String namePart) {
        return (root, query, cb) -> {
            if (namePart == null || namePart.isEmpty()) {
                return null;
            }
            return cb.like(cb.lower(root.get("name")), "%" + namePart.toLowerCase() + "%");
        };
    } 


   /**
     * Creates a {@link Specification} to search {@link MetadataEntity}
     * by text in name or description.
     *
     * @param searchText Text to search for, or null.
     * @return A {@link Specification} matching name or description.
     */
    @NonNull
    public static Specification<MetadataEntity> hasSearchText(String searchText) {
        return (root, query, cb) -> {
            if (searchText == null || searchText.trim().isEmpty()) {
                return null;
            }

            String pattern = "%" + searchText.toLowerCase() + "%";

            return cb.or(
                    cb.like(cb.lower(root.get("name")), pattern),
                    cb.like(cb.lower(root.get("description")), pattern)
            );
        };
    }

}
