package com.finconsgroup.itserr.marketplace.metadata.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.SecurityRoles;
import com.finconsgroup.itserr.marketplace.metadata.dm.authorization.HasRoles;
import com.finconsgroup.itserr.marketplace.metadata.dm.authorization.RequireMetadataOwnership;
import com.finconsgroup.itserr.marketplace.metadata.dm.authorization.RequireMetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputCreateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputInternalFindMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.InputUpdateMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.MetadataStatus;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataFieldExtDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.dto.OutputMetadataPreviewDto;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.ArchivedMetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.MetadataFieldEntity;
import com.finconsgroup.itserr.marketplace.metadata.dm.entity.enumerated.MetadataCategoryEnum;
import com.finconsgroup.itserr.marketplace.metadata.dm.exception.MetadataExistsException;
import com.finconsgroup.itserr.marketplace.metadata.dm.exception.MetadataNotFoundException;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.ArchivedMetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.MetadataFieldMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.ArchivedMetadataRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.MetadataFieldRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.MetadataRepository;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification.MetadataFieldSpecifications;
import com.finconsgroup.itserr.marketplace.metadata.dm.repository.specification.MetadataSpecifications;
import com.finconsgroup.itserr.marketplace.metadata.dm.service.MetadataService;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * Default implementation of {@link MetadataService} to perform operations related
 * to metadata resources
 */
@Service
@RequiredArgsConstructor
public class DefaultMetadataService implements MetadataService {

    /**
     * Metadata repository.
     */
    private final MetadataRepository metadataRepository;

    /**
     * MetadataFieldRepository repository.
     */
    private final MetadataFieldRepository metadataFieldRepository;

    /**
     * ArchivedMetadata repository.
     */
    private final ArchivedMetadataRepository archivedMetadataRepository;

    /**
     * Metadata mapper.
     */
    private final MetadataMapper metadataMapper;

    /**
     * MetadataFieldMapper mapper.
     */
    private final MetadataFieldMapper metadataFieldMapper;

    /**
     * ArchivedMetadata mapper.
     */
    private final ArchivedMetadataMapper archivedMetadataMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetadataPreviewDto> findAll(MetadataCategoryEnum metadataCategoryEnum, Optional<UUID> userId, @NonNull Pageable pageable) {
        Specification<MetadataEntity> statusSpec = MetadataSpecifications.hasStatusIn(Set.of(MetadataStatus.APPROVED));
        Specification<MetadataEntity> visibilitySpec;
        if (userId.isPresent() && (metadataCategoryEnum == null || metadataCategoryEnum.equals(MetadataCategoryEnum.PERSONAL))) {
            visibilitySpec =
            MetadataSpecifications.excludeCategory(MetadataCategoryEnum.PERSONAL)
            .or(
                MetadataSpecifications.hasCategory(MetadataCategoryEnum.PERSONAL)
                    .and(MetadataSpecifications.hasCreatorId(userId.get()))
            );
        }
        else {
            visibilitySpec = MetadataSpecifications.excludeCategory(MetadataCategoryEnum.PERSONAL);
        }
        visibilitySpec = visibilitySpec.and(MetadataSpecifications.hasCategory(metadataCategoryEnum));
        return metadataRepository.findAll(statusSpec.and(visibilitySpec), pageable)
                .map(metadataMapper::metadataEntityToMetadataPreviewDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputMetadataFieldDto> findAllFieldsById(UUID metadataId, @NonNull Pageable pageable) {
        return metadataFieldRepository.findAllByMetadataId(metadataId, pageable)
                .map(metadataFieldMapper::metadataFieldEntityToMetadataFieldDto);
    }

    @NonNull
    @Override
        @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
        /**
         * Retrieves a paginated list of all metadata fields, filtered by category and user visibility.
         * <p>
         * - Only fields belonging to APPROVED metadata are returned.
         * - If a user is present and the category is null or PERSONAL, PERSONAL fields are included only if created by the user.
         * - Otherwise, PERSONAL fields are excluded.
         * </p>
         *
         * @param category the metadata category to filter by, or null for all
         * @param pageable the pagination information (must not be null)
         * @return a page of OutputMetadataFieldExtDto representing the metadata fields
         */
        public Page<OutputMetadataFieldExtDto> findAllFields(MetadataCategoryEnum category, Pageable pageable) {
        // Ensure pageable is not null
        Pageable nonNullPageable = Objects.requireNonNull(pageable, "pageable");

        // Get the current user (if any)
        Optional<UUID> userId = JwtTokenHolder.getUserId();

        // Only fields from APPROVED metadata
        Specification<MetadataFieldEntity> statusSpec = MetadataFieldSpecifications.metadataHasStatusIn(Set.of(MetadataStatus.APPROVED));

        // Visibility: exclude PERSONAL unless user is present and requesting PERSONAL (then only their own)
        Specification<MetadataFieldEntity> visibilitySpec;
        if (userId.isPresent() && (category == null || category.equals(MetadataCategoryEnum.PERSONAL))) {
            visibilitySpec =
                MetadataFieldSpecifications.metadataExcludeCategory(MetadataCategoryEnum.PERSONAL)
                    .or(
                        MetadataFieldSpecifications.metadataHasCategory(MetadataCategoryEnum.PERSONAL)
                            .and(MetadataFieldSpecifications.metadataHasCreatorId(userId.get()))
                    );
        } else {
            visibilitySpec = MetadataFieldSpecifications.metadataExcludeCategory(MetadataCategoryEnum.PERSONAL);
        }

        // Filter by category if provided
        Specification<MetadataFieldEntity> categorySpec = MetadataFieldSpecifications.metadataHasCategory(category);

        // Always fetch-join metadata to avoid N+1
        Specification<MetadataFieldEntity> fetchSpec = MetadataFieldSpecifications.fetchMetadata();

        // Compose all specs and query
        return metadataFieldRepository
            .findAll(fetchSpec.and(statusSpec).and(visibilitySpec).and(categorySpec), nonNullPageable)
            .map(metadataFieldMapper::metadataFieldEntityToMetadataFieldExtDto);
        }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputMetadataDto findById(@NonNull final UUID metadataId) {
        return metadataRepository.findById(metadataId)
                .map(metadataMapper::metadataEntityToMetadataDto)
                .orElseThrow(() -> new MetadataNotFoundException(metadataId));
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @HasRoles({SecurityRoles.MEMBER_ROLE})
    public OutputMetadataDto create(@NonNull final InputCreateMetadataDto request, @NonNull UUID userId) {

        // Initialize
        final String name = StringUtils.trim(request.getName());

        // Checks that there is no other metadata with the same name
        if (metadataRepository.countByNameIgnoreCase(name) > 0) {
            throw new MetadataExistsException(name);
        }

        // Map to entity
        final MetadataEntity metadata = metadataMapper.metadataSaveRequestDtoToMetadataEntity(request, userId);

        // Set status to draft
        metadata.setStatus(MetadataStatus.DRAFT);

        // Set updatedBy
        metadata.setUpdatedBy(userId);

        // Save
        MetadataEntity savedMetadata = metadataRepository.saveAndFlush(metadata);

        // Return DTO of the saved entity
        return metadataMapper.metadataEntityToMetadataDto(savedMetadata);

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @HasRoles({SecurityRoles.MEMBER_ROLE})
    @RequireMetadataOwnership()
    @RequireMetadataStatus(allowed = {MetadataStatus.DRAFT, MetadataStatus.REJECTED})
    public OutputMetadataDto update(@NonNull UUID metadataId, @NonNull InputUpdateMetadataDto inputUpdateMetadataDto, @NonNull UUID userId) {

        // Fetch the Metadata
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Metadata not found: %s", metadataId)));

        final String currentName = StringUtils.trimToNull(metadataEntity.getName());
        final String updatedName = StringUtils.trimToNull(inputUpdateMetadataDto.getName());

        if (!StringUtils.equalsIgnoreCase(currentName, updatedName)
                && metadataRepository.countByNameIgnoreCase(updatedName) > 0) {
            throw new MetadataExistsException(updatedName);
        }

        metadataMapper.updateEntity(inputUpdateMetadataDto, metadataEntity);

        // Set status to DRAFT
        metadataEntity.setStatus(MetadataStatus.DRAFT);

        // Set updatedBy
        metadataEntity.setUpdatedBy(userId);

        // Save the updated item
        metadataEntity = metadataRepository.saveAndFlush(metadataEntity);

        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, noRollbackFor = Exception.class, readOnly = true)
    public List<OutputMetadataDto> find(@Nullable final InputInternalFindMetadataDto request) {

        // Find metadata according to criteria
        final List<MetadataEntity> foundMetadata =
                request == null
                        ? metadataRepository.findAll()
                        : metadataRepository.findAll(
                        MetadataSpecifications
                                .ids(request.getId()));

        // Return DTOs of the found entities
        return foundMetadata.stream()
                .map(metadataMapper::metadataEntityToMetadataDto)
                .toList();

    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @HasRoles({SecurityRoles.MEMBER_ROLE})
    @RequireMetadataOwnership()
    public OutputMetadataDto deleteById(@NonNull final UUID metadataId) {
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new MetadataNotFoundException(metadataId));
        ArchivedMetadataEntity archivedMetadataEntity = archivedMetadataMapper.metadataEntityToArchivedMetadataEntity(metadataEntity);
        archivedMetadataRepository.save(archivedMetadataEntity);
        metadataRepository.delete(metadataEntity);
        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @HasRoles({SecurityRoles.MEMBER_ROLE})
    @RequireMetadataOwnership()
    @RequireMetadataStatus(allowed = {MetadataStatus.DRAFT})
    public OutputMetadataDto requestModeration(UUID metadataId, UUID userId) {
        // Fetch the Metadata
        MetadataEntity metadataEntity = metadataRepository.findById(metadataId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(String.format("Metadata not found: %s", metadataId)));

        // Set metadata status to pending
        metadataEntity.setStatus(MetadataStatus.PENDING);
        metadataEntity.setModerationMessage(null);

        // Set updatedBy
        metadataEntity.setUpdatedBy(userId);

        metadataEntity = metadataRepository.saveAndFlush(metadataEntity);
        return metadataMapper.metadataEntityToMetadataDto(metadataEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, noRollbackFor = Exception.class, readOnly = true)
    public OutputPageDto<OutputMetadataDto> findAllByCreatorId(MetadataStatus status, @NonNull UUID creatorId, @NonNull Pageable pageable) {
        Set<MetadataStatus> statusList =
                status != null
                        ? Set.of(status)
                        : Set.of(MetadataStatus.APPROVED, MetadataStatus.DRAFT, MetadataStatus.PENDING, MetadataStatus.REJECTED);

        Specification<MetadataEntity> spec = MetadataSpecifications.hasStatusIn(statusList)
                .and(MetadataSpecifications.hasCreatorId(creatorId));

        return OutputPageDto.fromPage(metadataRepository.findAll(spec, pageable)
                .map(metadataMapper::metadataEntityToMetadataDto));
    }
}
